; Various SIL opcode routines
; Written by Viktors Berstis

        .data
        align   16
SR1     DQ      0
SR2     DQ      0
SR3     DQ      0
        .code

SIL18   PROC      ; X<12> CPYPAT Copy pattern
        MOV     SR1,r8          ;A1
        MOV     SR2,r9          ;A2
        ;       r10             ;A3
        ;       r11             ;A4
        ;       r12             ;A5
        MOV     SR3,r13         ;A6
SIL18L:
        MOV     rsi,SR2
        MOV     rdi,SR1

        MOV     rax,[rsi+DESCR]
        MOV     [rdi+DESCR],rax
        MOV     rax,[rsi+DESCR+VOFFSET]
        MOV     [rdi+DESCR+VOFFSET],rax ;COPY A7 F7 V7
        SHR     rax,8           ;get V7
        MOV     rdx,rax         ;save V7

        CMP     rax,3           ;MORE IF V7=3
        JNE     SIL18NX
        MOV     rax,[rsi+DESCR*4]
        MOV     [rdi+DESCR*4],rax
        MOV     rax,[rsi+DESCR*4+VOFFSET]
        MOV     [rdi+DESCR*4+VOFFSET],rax ;COPY A10 F10 V10
SIL18NX:

        MOV     rax,[rsi+DESCR*2] ;A8
        AND     rax,rax
        JZ      SIL18F1Z
        ADD     rax,r11
SIL18F1Z: MOV   [rdi+DESCR*2],rax    ;F1(A8)

        MOV     rax,[rsi+DESCR*2+VOFFSET]    ;V8
        SHR     rax,8
        AND     rax,rax
        JZ      SIL18F2Z
        ADD     rax,r11
        JMP     SIL18F2E
SIL18F2Z: MOV   rax,r12
SIL18F2E:
        SHL     rax,8
        MOV     [rdi+DESCR*2+VOFFSET],rax    ;F2(V8)

        MOV     rax,[rsi+DESCR*3]           ;A9
        ADD     rax,r10                     ;+A3
        MOV     [rdi+DESCR*3],rax           ;A9+A3

        MOV     rax,[rsi+DESCR*3+VOFFSET]
        SHR     rax,8                       ;V9
        ADD     rax,r10                     ;+A3
        SHL     rax,8
        MOV     [rdi+DESCR*3+VOFFSET],rax   ;V9+A3

        INC     rdx             ;V7 + 1
        SHL     rdx,4           ;multiply by DESCR ###
        ADD     SR1,rdx         ;r1=r1+(1+V7)*DESCR
        SUB     SR3,rdx         ;r3=r3-(1+V7)*DESCR
        JC      SIL18D  ;DONE
        JZ      SIL18D  ;DONE
        ADD     SR2,rdx         ;r2=r2+(1+V7)*DESCR
        JMP     SIL18L

SIL18D:                         ;SET FINAL R1 VALUE
        MOV     rax,SR1
        ret
SIL18   ENDP

; DATE() function
; Return date in 22 character string of form: "2000-02-28 19:40:00.18"
; (time is for current time zone)              yyyy mm dd hh mm ss cc
; yyyy = year
; mm = month
; dd = day of month
; hh = hour in 24 hour mode
; mm = minute
; ss = second
; cc = centiseconds
SIL19   PROC      ; X<13> DATE - returns time and date
ifdef linuxenvironment          ;LINUX version ===============================
; both of these return UTC time:
;       mov     rax,96          ;sys_gettimeofday
        mov     rax,201         ;sys_time
        lea     rdi,lnxtime     ; struct __kernel_old_timeval __user *tv
        xor     rsi,rsi         ; time zone adjustment not returned any more
;lea rsi,lnxtimezone ;#### apparently we cant get time zone offset here any more
        syscall

        mov     rax,lnxtime     ;UNIX time
        add     rax,lnxltmadjust ;adjust to local time

        call    ut2date

else                            ;WINDOWS version =============================
        externdef GetLocalTime : near
        winstack
        lea     rcx,wintime
        call    GetLocalTime
        winstacke
        xor     rax,rax
        lea     rdi,timetext+3
        mov     rcx,4
        mov     ax,wintime
        call    lilcvt
        lea     rdi,timetext+6
        mov     rcx,2
        mov     ax,winmonth
        call    lilcvt
        lea     rdi,timetext+9
        mov     rcx,2
        mov     ax,windayofmonth
        call    lilcvt
        lea     rdi,timetext+12
        mov     rcx,2
        mov     ax,winhour
        call    lilcvt
        lea     rdi,timetext+15
        mov     rcx,2
        mov     ax,winminute
        call    lilcvt
        lea     rdi,timetext+18
        mov     rcx,2
        mov     ax,winsecond
        call    lilcvt
        lea     rdi,timetext+22
        mov     rcx,3
        mov     ax,winmillisecond
        call    lilcvt
endif
        ret

SIL19   ENDP

; lilcvt - little convert from binary to decimal ascii
; rdi - address of least sig digit target
; rcx - number of digits desired
; rax - number to convert
lilcvt  PROC
        push    rdx
lilcvtl: xor     rdx,rdx
        div     ten
        add     dl,'0'
        mov     [rdi],dl
        dec     rdi
        loop    lilcvtl
        pop     rdx
        ret
lilcvt  ENDP

; convert unix time in rax to date string in timetext of length 23
ut2date PROC
        add     rax,31536000    ;seconds in a year
        mov     r8,rax          ;r8 = ut (to make leap year be correct remainder)
                                ;this won't work in year for multiples of 400 years
        xor     rdx,rdx
        mov     rbx,126230400   ;seconds in 4 years
        div     rbx
        mov     r9,rax          ;r9 = fourys = number of 4 years blocks
        mov     r10,rdx         ;r10 = fouryxs = number secs in 4 year block

        mov     rax,r10
        xor     rdx,rdx
        mov     rbx,31536000    ;seconds in a year
        div     rbx
        mov     r11,rax         ;r11 = xyears = 0 to 4 years within 4 year block
        mov     r12,rdx         ;r12 = siny = seconds within the year

        cmp     r11,4
        jne     @F
        dec     r11             ;r11 now 0-3
        add     r12,31536000    ;r12 = siny = seconds within year adjusted for leap

@@:     mov     rax,r12         ;r12
        xor     rdx,rdx
        mov     rbx,86400       ;seconds in a day
        div     rbx
        mov     r13,rax         ;r13 = days within the year

        mov     rax,r9
        add     rax,r9
        add     rax,r9
        add     rax,r9
        add     rax,r11
        add     rax,1969
        mov     r8,rax          ;r8 = year number

        lea     rdi,timetext+3  ;year text
        mov     rcx,4
        call    lilcvt

        mov     rax,r8          ;r8 year
        mov     rbx,4
        xor     rdx,rdx
        div     rbx
        cmp     rdx,0           ;is it a leap year?
        jz      @F
        mov     rdx,2
@@:     mov     r8,rdx          ;r8 = 0=leap, 2=not leap

        lea     rsi,mdays       ;days tables
        add     rsi,r8          ;select proper days table per leap year
        mov     rbx,r13         ;r13 days within the year
        xor     rcx,rcx         ;will be the month number

@@:     mov     ax,[rsi]        ;get month day count
        cmp     bx,ax
        jl      @F
        mov     r8,rax          ;r8 = prior days in year to this month
        inc     rcx
        add     rsi,4           ;next table entry
        jmp     @B
@@:
        mov     rax,rcx         ;month number  #### does this need inc?
        mov     rcx,2
        lea     rdi,timetext+6  ;month text
        call    lilcvt

        mov     rax,r13         ;r13 day within the year
        inc     rax
        sub     rax,r8          ;r8 = day of the month

        mov     rcx,2
        lea     rdi,timetext+9  ;day of month text
        call    lilcvt


        mov     rax,r12         ;seconds within the year
        mov     rbx,86400       ;seconds in a day
        xor     rdx,rdx
        div     rbx
        mov     rax,rdx         ;seconds within the day
        mov     rbx,3600        ;seconds in an hour
        xor     rdx,rdx
        div     rbx

        mov     rcx,2
        lea     rdi,timetext+12 ;hours rax has hours
        call    lilcvt

        mov     rax,rdx         ;seconds into hour
        xor     rdx,rdx
        mov     rbx,60          ;seconds in a minute
        div     rbx
        mov     rcx,2
        lea     rdi,timetext+15 ;minutes
        call    lilcvt

        mov     rax,rdx
        mov     rcx,2
        lea     rdi,timetext+18 ;seconds
        call    lilcvt
        ret
ut2date ENDP


SIL22   PROC      ; X<16> DIVIDE Divide integers
; input RAX to be divided by RCX,  RCX is not zero
; indicate problem with Overflow flag
; put quotient in RAX, remainder in RDX
        xor     rdx,rdx
        mov     DIVSIGN,0
        cmp     rcx,0
        jnl     SIL22nlc
        neg     rcx
        jo      SIL22ret
        XOR     DIVSIGN,1       ;INVERT RESULT SIGN
SIL22nlc:
        cmp     rax,0
        jnl     SIL22nla
        neg     rax
        jo      SIL22ret
        XOR     DIVSIGN,1       ;INVERT RESULT SIGN
SIL22nla:
        div     rcx
        cmp     DIVSIGN,1
        jne     SIL22ret
        neg     rax
SIL22ret: ret
SIL22   ENDP

SIL24   PROC    ;exit process
ifdef customaddon
; more custom for for Personal Editor 3
        cmp     pe3called,0
        je      pe3notcalled
        mov     q3todo,99 ; get curses to close so terminal does not lock up
ifdef linuxenvironment
        externdef s5cur : near
        lea     rdi,pe3data
        call    s5cur
else
        externdef s5cur : near
        winstack
        lea     rcx,pe3data
        call    s5cur
        winstacke
endif
pe3notcalled:
endif

        mov     rax,ABNDCL              ;&ABEND value
        cmp     rax,0
        jle     SIL24nomsg
        call    dbgprtfi
        call    dbgs
        db      ' = &ABEND Abnormal End to program',13,10,0
SIL24nomsg:
        cmp     rax,RETCOD
        jge     @F
        mov     rax,RETCOD      ;Use max of &CODE and &ABEND
@@:     cmp     rax,0
        jge     SIL24notneg     ;consider a negative code as zero
        xor     rax,rax         ;zero
SIL24notneg:
ifdef linuxenvironment
        mov     rdi,rax             ;return code
        cmp     rdi,255
        jna     abrcok
        mov     rdi,255             ; highest return code allowed
abrcok: mov     rax,60              ;exit call
        syscall                     ;exit program
else
        externdef ExitProcess : near
        winstack
        mov     rcx,rax             ; return code
        cmp     rcx,2147483647
        jna     abrcok
        mov     rcx,2147483647      ; highest return code allowed
abrcok: call    ExitProcess
endif
 dbgustr 'ExitProcess failed ####'
        xor     rax,rax
        push    rax
        ret                         ;or just crash
SIL24   ENDP

;SIL25  PROC                            ;ENFILE - end of file
        include ioclos.inc

;*       EXPINT    &CL1,&CL2,&CL3,&FLOC,&SLOC
;EXPINT= PUT(LABELCO TAB 'mov rcx,' OP3)
;        PUT(TAB 'mov rax,' OP2)
;        PUT(TAB 'call sil26')
;        DIFFER(OP4) PUT(TAB 'jc ' OP4)
;        PUT(TAB 'mov ' OP1 ',rax')
;        DIFFER(OP5) PUT(TAB 'jnc ' OP5)                         ;(STMT)
SIL26   PROC      ;X<1A> EXPINT Exponentiate integers x^y
        cmp     rcx,0           ;exponent zero? or negative?
        jl      sil26negx       ;negative->don't create fraction.. return zero or 1
        jg      sil26xn0        ;exponent postive
        cmp     rax,0           ;is x zero too?
        je      sil26bad        ;indeterminate per mathematica
        mov     rax,1           ;return 0
        clc
        ret

sil26negx:                      ;negative exponent case
        cmp     rcx,-1
        jne     sil26moreneg    ;exponent not -1
        cmp     rax,1           ; 1^(neg) = 1 (-1)^(-1) = -1
        je      sil26ok         ;return 1
        cmp     rax,-1
        je      sil26ok         ;return -1
sil26zero:
        xor     rax,rax         ;otherwise return zero
        clc
        ret

sil26one:
        mov     rax,1
        clc
        ret

sil26moreneg:                   ;exponent is less than -2
        cmp     rax,1           ;is x=1
        je      sil26ok         ;return x
        cmp     rax,-1          ;is x=-1
        jne     sil26zero
        test    rcx,1           ;is negative exponent even or odd?
        jz      sil26one        ;return one for even
        mov     rax,-1          ;return -1 for odd
        clc
        ret

sil26xn0:                       ;positive exponenet
        cmp     rax,0
        je      sil26ok         ; 0**y is zero
        cmp     rcx,1
        je      sil26ok         ; x**1 is x
        cmp     rax,0           ; is x negative?
        jl      sil26neg
        mov     rbx,rax
        dec     rcx
sil26lp:
        mul     rbx
        and     rdx,rdx
        jnz     sil26bad        ;fail if overflows
        test    rax,SIL100highbit
        jnz     sil26bad        ;fail if overflows to sign bit
        loop    sil26lp
sil26ok: clc
        ret

sil26neg:
        neg     rax
        jo      sil26bad        ;could not negate max negative number
        cmp     rax,1           ;was x -1?
        je      sil26xp1
        mov     r8,1            ;indicate sign
        mov     rbx,rax
        dec     rcx
sil26lp2:
        mul     rbx
        and     rdx,rdx
        jnz     sil26bad        ;fail if overflows
        xor     r8,1            ;flip sign bit
        loop    sil26lp2
        and     r8,r8
        jz      sil26ok         ;positive result
        neg     rax             ;make result negative
        clc
        ret

sil26xp1:                       ;expoenetiating -1 to positive exponent
        test    rcx,1           ;is exponenent even or odd?
        jz      sil26ok         ;return the one in rax
        neg     rax
        clc
        ret

sil26bad: stc
        ret
SIL26   ENDP


; GETBAL
; rdi = address of OP1 SPEC
; rcx = N from OP2 A field
; OUTPUT:
; update L field of SPEC
; carry flag on failure
SIL30   PROC      ; X<1E> GETBAL Get shortest non-null parenthesis balanced string
        mov     rsi,[rdi]               ;A
        mov     rdx,QWORD PTR [rdi+LOFFSET]     ;L
        shr     rdx,8
        add     rsi,QWORD PTR [rdi+OOFFSET]     ;A+O address in rsi
        add     rsi,rdx                 ;CL+1 character address
        lea     r8,systcommand
        xor     r9,r9                   ;depth counter
        xor     rbx,rbx                 ;J-1

;rsi is address of CL+1
;rdx = L
;rcx = N
;rbx = J-1
        mov     al,[rsi+rbx]
        cmp     al,'('
        je      SIL30go1
        cmp     al,'['
        je      SIL30go
        cmp     al,'{'
        je      SIL30go
        cmp     al,')'
        je      SIL30f
        cmp     al,']'
        je      SIL30f
        cmp     al,'}'
        je      SIL30f
        inc     rdx                     ;L+1
        shl     rdx,8
        mov     qword ptr [rdi+LOFFSET],rdx
        jmp     SIL30s

SIL30go1: dec   al
SIL30go: add    al,2
        inc     r9                      ;increment depth
        mov     [r8+r9],al              ;save matching right side character
        cmp     r9,systcommandl
        jnl     SIL30f                  ;fail if way too deep
SIL30nx:
        inc     rbx                     ;next char
        dec     rcx                     ;decrement N
        jz      SIL30f                  ;ran out of characters

        mov     al,[rsi+rbx]
        cmp     al,'('
        je      SIL30go1
        cmp     al,'['
        je      SIL30go
        cmp     al,'{'
        je      SIL30go
        cmp     al,')'
        je      SIL30up
        cmp     al,']'
        je      SIL30up
        cmp     al,'}'
        je      SIL30up
        jmp     SIL30nx

SIL30up:
        cmp     al,[r8+r9]              ;does it match left side?
        jne     SIL30f
        dec     r9                      ;decrement depth
        jnz     SIL30nx
; Done!
        inc     rbx                     ;resulting J
;showreg rbx
        shl     rbx,8
        add     qword ptr [rdi+LOFFSET],rbx
SIL30s: clc
        ret
SIL30f: stc                             ;set carry to indicate failure
        ret
SIL30   ENDP

;LEXCMP= PUT(LABELCO TAB 'mov rsi,' OP1)
;        PUT(TAB 'add rsi,qword ptr [' OP1 '+OOFFSET]')
;        PUT(TAB 'mov rcx,qword ptr [' OP1 '+LOFFSET]')
;        PUT(TAB 'mov rdi,' OP2)
;        PUT(TAB 'add rdi,qword ptr [' OP2 '+OOFFSET]')
;        PUT(TAB 'mov rdx,qword ptr [' OP2 '+LOFFSET]')
;        PUT(TAB 'call SIL45')
;        DIFFER(OP3) IDENT(OP3,OP5) PUT(TAB 'jne ' OP3)
;        DIFFER(OP4) PUT(TAB 'je ' OP4)
;        DIFFER(OP3) DIFFER(OP3,OP5) PUT(TAB 'ja ' OP3)
;        DIFFER(OP5) DIFFER(OP3,OP5) PUT(TAB 'jb ' OP5)          :(STMT)
SIL45   PROC            ;LEXCMP
; string1 is at rsi for length rcx
; string2 is at rdi for length rdx
SIL45lp:
        and     rcx,rcx
        jnz     SIL45o1nn
        and     rdx,rdx
        jnz     SIL45o2nn
        ret                             ;return with equal for two null strings
SIL45o2nn: cmp  rcx,rdx
        ret                             ;string1 is less
SIL45o2low: cmp rdx,rcx
SIL45ret: ret
SIL45o1nn: and  rdx,rdx
        jz      SIL45o2low              ;string2 is less

        mov     al,[rsi]
        cmp     al,[rdi]
        jnz     SIL45ret                ;quit if different
        dec     rcx
        dec     rdx
        inc     rsi
        inc     rdi
        jmp     SIL45lp
SIL45   ENDP


;#### SIL46    PROC             ;LINK
;####         CALL    DBGS
;####         DB      'SIL46 not implemented ####',13,10,0
;####         call DosExit
;#### SIL46    ENDP
;####

;LINKOR= PUT(LABELCO TAB 'mov rsi,' OP1)
;        PUT(TAB 'mov rdx,' OP2)
;        PUT(TAB 'call SIL47')                                   :(STMT)
SIL47   PROC
        xor     rbx,rbx
        add     rsi,DESCR*2
SIL47lp: mov    rax,rbx
        mov     rbx,[rsi+rax]
        cmp     rbx,0
        jnz     SIL47lp
        mov     [rsi+rax],rdx
        ret
SIL47   ENDP


;#### SIL48    PROC             ;LOAD
;####         CALL    DBGS
;####         DB      'SIL48 not implemented ####',13,10,0
;####         call DosExit
;#### SIL48    ENDP
;####

SIL49   PROC      ; X<32> LOCAPT Locate attribute pair by value
        mov     rax,[r10]
        mov     rbx,[r10+VOFFSET]       ;rax:rbx is what we are searching for
        mov     rsi,[r9]                ;address in descr2
        mov     rdx,[rsi+VOFFSET]
        shr     rdx,8
        add     rdx,rsi                 ;last address to look at
        add     rsi,DESCR               ;first address to look at
SIL49lp: cmp    rsi,rdx
        ja      SIL49fail               ;reached end
        cmp     rax,[rsi]
        jne     SIL49nxt
        cmp     rbx,[rsi+VOFFSET]
        jne     SIL49nxt
        sub     rsi,DESCR               ;address to return
        mov     [r8],rsi
        xor     rax,rax
        mov     [r8+VOFFSET],rax        ;s370 sets this to zero, so we will too
        cmp     rax,rax                 ;set equal flags
        ret
SIL49nxt: add   rsi,DESCR+DESCR
        jmp     SIL49lp
SIL49fail:
        cmp     rsi,1                   ;set flags to not equal
        ret
SIL49   ENDP

SIL63   PROC    ; ================ MSTIME - get nanosecond time ===============
        storxptr
; none of these versions were perfect, cycleversion had the fewest problems
        jmp     cycleversion
;####   jmp     cpuversion
;####   jmp     wallversion

; Following is wall clock version:
wallversion:
ifdef linuxenvironment
        mov     rax,228 ;sys_clock_gettime system call
        mov     rdi,0   ;CLOCK_REALTIME
        lea     rsi,timeplace
        syscall
        mov     rax,qword ptr [timeplace]       ;seconds
        sub     rax,1640995200  ;make it start at 2022/01/01 (good for >68 years ###)
        mov     rbx,1000000000  ;make seconds into nanoseconds
        mul     rbx
        add     rax,qword ptr [timeplace+8]     ;add in nanoseconds part
else
; windows version here
        externdef GetSystemTimeAsFileTime : near
        winstack                    ; must leave 32 bytes space plus for more parameters
        lea     rcx,timeplace
        call    GetSystemTimeAsFileTime
        mov     rax,qword ptr [timeplace]       ;in 100 nanosecond units
        sub     rax,winzerotime                 ;start point is 2022/01/01
        mov     rbx,100         ;convert to nanoseconds to match Linux
        mul     rbx
        winstacke                   ; must leave 32 bytes space plus for more parameters
endif
        xor     rbx,rbx
        loadxptr
        ret

cpuversion:                             ; =====================================
; Following is the cpu time version:  (not working correctly)
ifdef linuxenvironment
CLOCK_PROCESS_CPUTIME_ID equ 2
 dbgustr 'cycle count:'
 rdtsc
 showreg rdx
 showreg rax
 shl rdx,32
 showreg rdx
 and rax,0ffffffffh
 or  rdx,rax
 showreg rdx
 mov rax,rdx
 showreg rax
 call dbgprtfi
 call dbgcrlf
        mov     rax,64h ;sys_times system call
        lea     rdi,timeplaces
        syscall
 dbgustr 'sys_times call:'
 showreg rax
 call dbgprtfi
 lea rsi,timeplaces
 mov rcx,80
 call dbgdump
        mov     rax,qword ptr [timeplaces]      ;tms_utime
        mov     rbx,10000000  ;make hundredths into nanoseconds
        mul     rbx
else
; windows version here
        externdef GetCurrentProcess : near
        externdef GetCurrentThread : near
        externdef GetProcessTimes : near
        externdef GetThreadTimes : near
        externdef QueryProcessCycleTime : near
        winstack 1                  ; must leave 32 bytes space plus for more parameters
        call    GetCurrentProcess
        mov     rcx,rax         ; 1: process handle
        lea     rdx,timeplace   ; 2: creation time
;       lea     r8,junkplace    ; 3: exit time
;       lea     r9,timeplace    ; 4: kernel time
;       lea     rax,junkplace
;       mov     [rsp+32],rax    ; 5: user time
;####   call    GetProcessTimes
;####   call    GetThreadTimes
        call    QueryProcessCycleTime
        cmp     rax,0
        jnz     @F
        call    dbggle
        dbgustr 'error from GetProcessTimes'
@@:
        mov     rax,qword ptr [timeplace]       ;in 100 nanosecond units
;       sub     rax,winzerotime                 ;start point is 2022/01/01
        mov     rbx,100         ;convert to nanoseconds to match Linux
        mul     rbx
        winstacke                   ; must leave 32 bytes space plus for more parameters
endif
        xor     rbx,rbx
        loadxptr
        ret

cycleversion:                             ; =====================================
; Following is the cycle count version:
ifdef linuxenvironment
        call    lnxmhz
else
; windows version here
        externdef CallNtPowerInformation : near
        winstack
        mov     rcx,11          ; processor information
        mov     rdx,0
        mov     r8,0
        lea     r9,timeplaces
        mov     qword ptr [rsp+32],3072
        call    CallNtPowerInformation
        winstacke
        mov     rcx,64          ; max number of processors
        xor     r15,15          ; max value goes here
        xor     r14,14          ; offset
        lea     r14,dword ptr timeplaces+8  ;get current mhz processor speed
sil63pmlp:
        mov     eax,[r14]
        add     r14,24          ;skip to next processor
        cmp     eax,81818181h   ; are we at the end
        je      s63allscanned      ;all scanned
        cmp     rax,r15
        jna     @F
        mov     r15,rax         ;new max
@@:     loop    sil63pmlp
s63allscanned:
        mov     processormhz,r15
endif
        rdtsc
        shl     rdx,32
        and     rax,0ffffffffh
        or      rdx,rax
        mov     rax,rdx
;;      xor     randx,rax       ;#### random number seed
        xor     rbx,rbx
        loadxptr
        ret

SIL63   ENDP

SIL64   PROC      ; X<40> Multiply integers

SIL64NFLT:  ; X<40> Multiply integers
; Multiply RBX by RCX and put result at RDI

        MOV     r9,0    ;Assume positive final result
        AND     RCX,RCX ;TEST SIGN
        JNL     SIL64P1
        NEG     RCX
        JO      SIL64F
        MOV     r9,1    ;Indicate NEGATIVE RESULT
SIL64P1:

        AND     RBX,RBX ;TEST SIGN
        JNL     SIL64P2
        NEG     RBX
        JO      SIL64F
        XOR     r9,1    ;Invert sign of final result
SIL64P2:

        MOV     RAX,RBX
        MUL     RCX

        CMP     RDX,0
        JNE     SIL64F  ;Overflow
        CMP     RAX,SIL100maxpos
        JA      SIL64F  ;Overflow

        CMP     r9,0    ;INVERT RESULT?
        JE      SIL64NI
        NEG     RAX
        JO      SIL64F
SIL64NI: MOV    [rdi],rax
        xor     r8,r8   ;Success
        ret
SIL64F:
        mov     r8,1    ;Failure
        ret
SIL64   ENDP


;*       OUTPUT    &UNIT,&FORMAT,&LIST
;OUTPUT=
;        PUT(LABELCO TAB 'mov r8,' OP1)
;        PUT(        TAB 'lea r9,' OP2)
;* handle list
;outputlp1 OP3 ANY('()') = :S(outputlp1)
;        OP3 = DIFFER(OP3) OP3 ','
;        outputcnt = 10
;outputlp2
;        OP3 POS(0) BREAK(',') . OP3x ',' = :F(outputlp2e)
;        PUT(TAB 'lea r' outputcnt ',' OP3x)
;        outputcnt = outputcnt + 1 :(outputlp2)
;outputlp2e
;        PUT(TAB 'mov rdx,' outputcnt - 10)
;        PUT(TAB 'call SIL67')                                   :(STMT)
; r8 = io unit number
; r9 = address of format string
; r10... = substitution parameters
; rdx = count of substitution parameters
SIL67   PROC                    ;OUTPUT - output formatted message
; dbgustr 'at sil67 start'
        mov     rsi,r9
        lea     rdi,foutputbuf
sil67lp: mov    al,[rsi]
        cmp     al,0            ;end of format?
        je      sil67end
        cmp     al,'\'          ;substitute char?
        jne     sil67norm
        inc     rsi
        mov     al,[rsi]        ;get next char
        cmp     al,0            ;end of format?
        je      sil67end
        cmp     al,'^'          ;is it a cr or lf?
        je      sil67crlf
        cmp     al,'#'          ;is it an integer?
        je      sil67int
        cmp     al,'F'          ;is it a float?
        je      sil67flt
        jmp     sil67norm       ;just print it we don't know what this is
sil67int:
        mov     rbx,[r10]       ;get the integer
        call    INT2STR
        dec     rdi
sil67shft:
        mov     r10,r11         ;shift list
        mov     r11,r12
        mov     r12,r13
        mov     r13,r14
        mov     r14,r15
        jmp     sil67nxt
sil67flt:
        push    rsi
        mov     rax,[r10]       ;get the float number
;       mov     rsi,rdi
        lea     rsi,sil67fltbuf
        call    FLT2STR

; shorten float to 6 significant digits
        lea     rsi,sil67fltbuf
        xor     r15,r15         ;zero indicates decimal point not yet seen
        xor     r14,r14         ;how many significant digits seen
sil67lp1:
        jrcxz   sil67runout
        mov     al,[rsi]
        cmp     al,'.'
        jne     @F
        inc     r15             ;say saw the decimal
@@:
        cmp     al,'1'
        jb      sil67ndig       ;not significant digit
        cmp     al,'9'
        ja      sil67ndig
                                ;encountered a significant digit
                                ;now count 6 of them and include a period
        inc     r14             ;first significan digits seen
sil67ndig2:
        mov     [rdi],al
        inc     rdi
        inc     rsi
        dec     rcx
        mov     al,[rsi]
        cmp     al,'.'          ;check for decimal
        jne     @F
        inc     r15
@@:     cmp     al,'0'
        jb      sil67ndig2
        cmp     al,'9'
        ja      sil67ndig2
        inc     r14
        cmp     r14,7
        jb      sil67ndig2
        and     r15,r15
        jz      sil67ndig2
        jmp     sil67runout

sil67ndig:
        mov     [rdi],al        ;copy to result area
        inc     rdi
        inc     rsi
        dec     rcx
        jmp     sil67lp1        ;loop until we find a significant digit

; finished chopping
sil67runout:
        dec     rdi             ;account for /F
        pop     rsi
        jmp     sil67shft
sil67crlf:
        inc     rsi
        mov     al,[rsi]
        cmp     al,0            ;end of format?
        je      sil67end
        cmp     al,'M'          ;carriage return?
        je      sil67cr
        cmp     al,'J'          ;line feed?
        je      sil67lf
sil67cr: mov    al,13
        jmp     sil67norm       ;see if -cr processing specified #####
sil67lf: mov    al,10
sil67norm:
        mov     [rdi],al
sil67nxt: inc     rdi
        inc     rsi
        jmp     sil67lp
sil67end:

        lea rsi,foutputbuf
        mov rcx,rdi
        sub rcx,rsi
        mov     r10,6           ;output to unit 6
        call SIL104
        ret
SIL67   ENDP



SIL85   PROC      ; X<55> RPLACE Replace characters
        mov     r10,rcx         ;save length
        lea     rdi,RPLTAB
        mov     rcx,256
        xor     rax,rax
SIL85lp0: mov   byte ptr [rdi+rax],al   ;clear translate table
        inc     al
        loop    SIL85lp0
;       lea     rdi,RPLTAB      ;rdi should still be here and rax zero
        mov     rcx,rbx
SIL85lp1: mov   al,byte ptr [r8]
        mov     bl,byte ptr [r9]
        mov     byte ptr [rdi+rax],bl   ;fill translate table
        inc     r8
        inc     r9
        loop    SIL85lp1
        mov     rcx,r10         ;length of string to process
        mov     rsi,rdx
SIL85lp2: mov   al,[rsi]
        mov     bl,[rdi+rax]
        mov     byte ptr [rsi],bl
        inc     rsi
        loop    SIL85lp2
        ret

SIL85   ENDP

;#### SIL87    PROC             ;RESETFI - only used in garbage collector
;####         CALL    DBGS
;####         DB      'SIL87 not implemented ####',13,10,0
;#### SIL87    ENDP
;####
;####

SIL100  PROC      ; X<64> SPCINT Convert specifier to integer
; input: rcx is length, rsi is string location
; returns: integer in rcx, overflow flag if overflow
        mov     SIL100sign,0    ;0 indicates positive or zero, 1=negative
        xor     rbx,rbx
        xor     rax,rax ;result accumulated here
;; skip leading white space ####
;SIL100lbl:
;        cmp     byte ptr [rsi],' '
;        je      SIL100slb
;        cmp     byte ptr [rsi],9        ;tab
;        jne     SIL100lbd
;SIL100slb:
;        inc     rsi
;        dec     rcx
;        jnz     SIL100lbl
;        jmp     SIL100zero
;SIL100lbd:
        mov     bl,[rsi]
        cmp     bl,'+'
        jne     SIL100np
        inc     rsi
        dec     rcx
        jrcxz   SIL100badxx     ;jump if only a sign character
        jmp     SIL100sd        ;jump if + sign done
SIL100badxx: jmp SIL100bad
SIL100np: cmp   bl,'-'
        jne     SIL100sd        ;jump if not - sign
        mov     SIL100sign,1
        inc     rsi
        dec     rcx
        jrcxz   SIL100badxx     ;jump if only a sign character
        jmp     SIL100sd

SIL100lz: inc   rsi
        dec     rcx
SIL100sd: jrcxz SIL100zero
        mov     bl,[rsi]
        cmp     bl,'0'
        je      SIL100lz        ;jump if leading zero

SIL100lp: cmp   bl,'9'
        ja      SIL100bad       ;non didgit character found
        sub     bl,'0'
        jb      SIL100bad       ;non didgit character found
        add     rax,rbx         ;contribute to result
        jo      SIL100ovx       ;overflow
        inc     rsi
        dec     rcx
        jrcxz   SIL100nm        ;jump if no more digits
        mul     SIL100ten
        jo      SIL100ov
        mov     bl,[rsi]
        jmp     SIL100lp
SIL100nm: cmp   SIL100sign,0
        je      SIL100pos
        xor     rax,SIL100allbits
        inc     rax
        test    rax,SIL100highbit
        jz      SIL100ov
        xor     rbx,rbx
        mov     rcx,rax
        ret
SIL100pos: test rax,SIL100highbit
        jnz     SIL100ov
        mov     rcx,rax
        ret
SIL100zero: xor al,al    ;clear overflow
SIL100ov:
        ret

SIL100ovx:                      ;handle max neg integer case
        cmp     rax,SIL100highbit
        jne     SIL100bad       ;keep overflow
        cmp     rcx,1
        jne     SIL100bad
        cmp     SIL100sign,1
        jne     SIL100bad
        mov     rcx,SIL100highbit ;return result
        jmp     SIL100zero

SIL100bad:      ;handle bad chars
        mov     al,127
        add     al,1    ;create overflow to show error
        ret

SIL100  ENDP


;SIL104  PROC      ; X<68> STPRNT String write
        include stprnt.inc
        include openwrit.inc


;SIL105  PROC      ; X<69> STREAD String read
        include stread.inc
        include openread.inc

;SIL106  PROC      ; X<6A> STREAM
        include stream.inc


SIL113  PROC      ; X<71> TRIMSP Trim blanks (and tabs) from specifier
; Modified from original:
; OP1 is subject string to be trimmmed
; OP2 is null if blanks+tabs, else trim characters set by OP2
; OP3 is minimum length of result
; On entry:
; r8 = address of OP1 string to trim
; r9 = address of OP2 optional charcters to trim
; r10 = address of OP3 trim to this minimum length

        MOV     rcx,[r8+LOFFSET] ;LEN from OP1
        SHR     rcx,8
        MOV     rsi,[r8+AOFFSET] ;A
        ADD     rsi,[r8+OOFFSET] ;A+O
        ADD     rsi,rcx         ;A+O+L (one char beyond end of string)
        jrcxz   SIL113NT        ;No trim if null string
        MOV     rdx,[r10+AOFFSET];Min length
        CMP     rcx,rdx
        JL      SIL113NT        ;No trim if below min in length
        cmp     rdx,0
        JL      SIL113Z         ;negative number = 0 for trim
        JMP     SIL113go
SIL113NT: ret

SIL113Z: SUB    rdx,rdx         ;Zero min length
SIL113go:
        MOV     rbx,[r9+LOFFSET] ;Length of trim chars
        SHR     rbx,8
        AND     rbx,rbx         ;Default for op2 null
        JZ      SIL113def       ;just use blanks & tabs / white space
        mov     rdi,[r9]
        add     rdi,[r9+OOFFSET] ;location of trim chars
        mov     r11,rdi         ;save location
        mov     r12,rbx         ;save length

; handle case where op2 is not null
SIL113LP2: DEC  rsi
        CMP     rcx,rdx         ;Check minimum length
        JNA     SIL113D
        MOV     AL,[rsi]        ;get character of op1
        MOV     rbx,r12         ;get op2 length
        mov     rdi,r11         ;get op2 string loc
SIL113LP3:
        DEC     rbx
        CMP     AL,[rdi+rbx]
        JE      SIL113SK
;### could use xlate table for large op2 strings
        AND     rbx,rbx
        JZ      SIL113D         ;No more trim set chars, none matched
        JMP     SIL113LP3       ;check more from trim set

SIL113SK: LOOP  SIL113LP2
        jmp     SIL113D

; handle simple white space case
SIL113def:
SIL113deflp: DEC rsi
        CMP     rcx,rdx   ;Check minimum length
        JNG     SIL113D   ;jump if done
        MOV     AL,[rsi]
        CMP     AL,' '
        JE      SIL113TB
        CMP     AL,9    ;TAB
        JNE     SIL113D
SIL113TB: LOOP  SIL113deflp
SIL113D:
        SHL     rcx,8
        MOV     [r8+LOFFSET],rcx ;adjust length to trimmed length
        ret
SIL113  ENDP



;#### SIL114    PROC            ;UNLOAD
;####         CALL    DBGS
;####         DB      'SIL114 not implemented ####',13,10,0
;####         call DosExit
;#### SIL114    ENDP
;####

;#### SIL120  PROC         ; TRAPCK
;####         ret             ;#### not done - ignore for now - should be in-line
;#### SIL120  ENDP
;####


SIL121  PROC      ; X<79> FILNAMI and FILNAMO Set file name for IO unit
; UNITNUM, FILENAME,  ATTRIBS,  FAILURE, SUCCESS are parameters
;  DESCR     SPEC      SPEC      ADDR     ADDR
;   RAX       R8        R9        C        NC
; RBX is WOPEN for output, ROPEN for input
        mov     CURUNIT,0       ;clear current unit poitner
; allocate i/o unit
        cmp     rax,1
        jl      sunoor
        cmp     rax,UNITARRAYMAX
        jnl     sunoor
        jmp     sunoir

sunoor:  dbgustr 'Unit number out of range'
sil121fail:
        stc
        ret

sil121failp:
        jmp sil121fail

sunoir:
        shl     rax,3
        lea     rsi,UNITARRAY
        add     rsi,rax     ;address in UNITARRAY
        mov     r10,[rsi]   ;see if already allocated
        cmp     r10,0
        jnz     sunoal      ;jump if already allocated

; allocate UNIT structure
        mov     rcx,UNITSIZE
        push    rsi
        push    r8
        push    r9
        call    myalloc         ;assumes area is zeroed
        pop     r9
        pop     r8
        pop     rsi
        mov     [rsi],rax
        mov     r10,rax
; set some defaults, snobol5.asm does this too
ifdef linuxenvironment
        mov     dword ptr UNITSTR.UNITFLAGS[r10],UNITNOCR+UNITNOEF   ;default for linux
else
        mov     dword ptr UNITSTR.UNITFLAGS[r10],UNITNOEF            ;default for windows
endif

; UNIT structure is allocated
sunoal: mov     CURUNIT,r10     ;save current unit pointer
        mov     dword ptr UNITSTR.UNITVALID[r10],UNITVALIDCODE
        mov     UNITSTR.UNITOPEN[r10],bl ;indicate if for input or output

; copy in file name
        lea     rdi,UNITSTR.UNITNAME[r10]
        cmp     r8,0
        jz      sil121fail
        mov     rsi,[r8]        ;A field of filename spec
        add     rsi,[r8+OOFFSET] ;Add any offset
        mov     rcx,[r8+LOFFSET] ;get file name length
        shr     rcx,8
        cmp     rcx,0
        jz      sil121fail
        cmp     rcx,UNITNAMEL
        jnl     sil121fail      ;make sure file name is not too long
        rep movsb ;copy file name to unit area
        mov     byte ptr [rdi],0 ;add a trailing null

; parse file attributes
        cmp     r9,0
        jz      sil121noparse
        mov     rsi,[r9]        ;A field of attributes spec
        add     rsi,[r9+OOFFSET] ;Add any offset
        mov     rcx,[r9+LOFFSET] ;get attributes string length
        shr     rcx,8
        and     rcx,rcx
        jz      sil121noparse
        cmp     rcx,UNITNAMEL
        jnl     sil121fail      ;make sure attributes string is not too long
        pstart  rsi,rcx,sil121failp
          ppos  0
          parbnos
            include fileattr.inc      ;parse file attribute flags
            plp
              pspan   whitespace,whitespacelen
            palt
              plen    0
            prp
          parbnoe
          prpos   0
        pend

sil121noparse:
        clc
        ret
SIL121  ENDP


SIL122  PROC
        include block.inc
SIL122  ENDP


; SEEK ********=================================================================
; Input:
; 2: r8  UNIT - descriptor containing unit number
; 3: r9  OFFSET - descriptor containing integer offset required
; 4: r11 TYPE - type of seek 0=absolute 1=relative to current 2=relative to end
; Output:
; 4: r11 NEWPOS - descriptor to receive new offset in file
; 5: c   FAIL address
; 6: nc  SUCCESS address
;
; Input:
;         ------------------------------
; UNIT    |   A1   |         |         |
;         ------------------------------
;         ------------------------------
; OFFSET  |   A2   |         |         |
;         ------------------------------
;         ------------------------------
; TYPE    |   A3   |         |         |
;         ------------------------------
; Output:
;         ------------------------------
; NEWPOS  |   POS  |         |         |
;         ------------------------------
sil127seek proc
        include ioseek.inc
sil127seek endp




; CENTER =======================================================================
; Parameters:
; r8    TARGET - 4 byte specifier offset for target
; r9    STRING - 4 byte specifier offset for source
; r10   PAD - 4 byte specifier offset for pad character
; +16
; Input:
;         --------------------------------------------------
; TARGET  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; STRING  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         --------------------------------------------------
; PAD     |   A3   |         |         |    O3   |    L3   |
;         --------------------------------------------------
; Output:
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   CL1   |
;         ----------------------------------------
SIL127CENTER:
        mov     rsi,[r9+AOFFSET]
        add     rsi,[r9+OOFFSET]
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8
        mov     rdi,[r8+AOFFSET]
        add     rdi,[r8+OOFFSET]
        mov     rdx,[r8+LOFFSET]
        shr     rdx,8
        mov     bl,' '          ;default pad character
        mov     r11,[r10+AOFFSET]
        add     r11,[r10+OOFFSET]
        mov     rax,[r10+LOFFSET]
        shr     rax,8
        jz      centerdp                ;jump if using default pad char
        mov     bl,byte ptr [r11]       ;get pad character from PAD
centerdp:
        xor     r10,r10
        cmp     rcx,rdx
        je      centernoleftpad
        mov     rax,rdx
        sub     rax,rcx                 ;total pad character count
        mov     r10,rax
        shr     rax,1                   ;divide by two
        sub     r10,rax                 ;how many to pad on right
        and     rax,rax
        jz      centernoleftpad
centerleftpad:
        mov     [rdi],bl
        inc     rdi
        dec     rax
        jnz     centerleftpad
centernoleftpad:
        rep movsb                       ;move source string
centerrightpad:
        and     r10,r10
        jz      centerdone
        mov     [rdi],bl
        inc     rdi
        dec     r10
        jnz     centerrightpad
centerdone:
        ret



; GC ===========================================================================
        include gc.inc


; SBIT =========================================================================
; Parameters:
;       SOURCE - r8
;       TARGET - r9
; Converts string from Source into bits string in Target
; Input:
;         ---\----------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L2) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
; where Cn are the bits representing the string S (source operand)
; The bits are not in any endian order.. leftmost is high order, rightmost is
; low order
; L2 should equal L1*BITSPA
SIL127SBIT:
        mov     rdx,[r8+LOFFSET]
        shr     rdx,8                   ;target length
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;source length
        mov     rsi,[r9]
        add     rsi,[r9+OOFFSET]        ;source address
        jecxz   SIL127SBITNULL
SIL127SBITBYTELP:
        mov     al,[rsi]        ;Get string byte
        mov     rbx,BITSPA      ;bits per address
SIL127SBITLP:
        rcl     al,1            ;look at high order bit first
        jc      SIL127SBITON
        mov     byte ptr [rdi],'0' ;bit off
        JMP     SIL127NXSBIT
SIL127SBITON:
        mov     byte ptr [rdi],'1' ;bit on
SIL127NXSBIT:
        inc     rdi             ;one higher order output position
        dec     rbx             ;decrement bit counter
        jnz     SIL127SBITLP
        inc     rsi             ;next byte of source string
        loop    SIL127SBITBYTELP
SIL127SBITNULL:
        ret
; GETENV =====================================================================
; Parameters:
;       SOURCE - r8
;       TARGET - r9
; Gets value for source environment variable and places it in buffer
; Input:
;         ---\----------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |        |         |         |    0    |         |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L1) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; BUFFER  |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
;         --------------------------------------------------
; TARGET  |BUFFER  |         |         |    0    |   L2    |
;         --------------------------------------------------
; BUFFER is an area where the environment variable value is placed.
; Its address and length are placed in the target specifier.
; L2 is the length of the environment variable's data.
; If the environment variable was not found, then L2 is set to zero.
SILGETENV:
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        mov     rsi,[r9]
        add     rsi,[r9+OOFFSET]        ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;source length
        push    r8
        push    rdi
        call    s5getenv                ;retrieve the environment variable's value
        pop     rdi
        pop     r8
        lea     rsi,envbuffer
        mov     [r8+AOFFSET],rsi        ;put buffer address into specifier
        mov     rcx,envbufferlen
        shl     rcx,8
        mov     [r8+LOFFSET],rcx        ;put length into specifier
        ret

; IBIT =========================================================================
; Parameters:
;       SOURCE - r9
;       TARGET - r8
; Converts integer or real into bits
; Input:
;         ------------------------------
; SOURCE  |   I    |         |         |
;         ------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   CL64  |
;         ----------------------------------------
; where Cn are the bits representing the integer
; The bits are not in any endian order.. leftmost is high order, rightmost is low order
; L2 should equal 64 (ALENG*BITSPA)
SIL127IBIT:
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        mov     rax,[r9]                ;Get integer or real
        mov     rcx,ALENG*BITSPA        ;number of bits
SIL127IBITLP:
        rcl     rax,1                   ;look at high order bit first
        jc      SIL127IBITON
        mov     byte ptr [rdi],'0' ;bit off
        inc     rdi             ;one higher order output position
        loop    SIL127IBITLP
        ret
SIL127IBITON:
        mov     byte ptr [rdi],'1' ;bit on
        inc     rdi             ;one higher order output position
        loop    SIL127IBITLP
        ret


; SHEX =========================================================================
; Parameters:
;       SOURCE - r8
;       TARGET - r9
; Converts string from Source into hex digit string in Target
; Input:
;         --------------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L2) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
; where Cn are the hex digits representing the string S (source operand)
; The hex digits are in the same order as the source, leftmost is high order,
; rightmost is low order
; L2 should equal L1*BITSPA/4
SIL127SHEX:
        mov     rdx,[r8+LOFFSET]
        shr     rdx,8                   ;target length
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;source length
        mov     rsi,[r9]
        add     rsi,[r9+OOFFSET]        ;source address
        jecxz   SIL127SHEXNULL
SIL127SHEXBYTELP:
        mov     al,[rsi]        ;Get string byte
        rol     al,4            ;look at high order part first
        mov     bl,al
        and     rbx,15
        cmp     rbx,10
        jl      SIL127SHEXlow
        add     rbx,'A'-10
        jmp     SIL127SHEX2
SIL127SHEXlow:
        or      bl,'0'
SIL127SHEX2:
        mov     byte ptr [rdi],bl
        inc     rdi
        rol     al,4            ;look at low order part
        mov     bl,al
        and     rbx,15
        cmp     rbx,10
        jl      SIL127SHEXhi
        add     rbx,'A'-10
        jmp     SIL127SHEX3
SIL127SHEXhi:
        or      bl,'0'
SIL127SHEX3:
        mov     byte ptr [rdi],bl
        inc     rdi
        inc     rsi             ;next byte of source string
        loop    SIL127SHEXBYTELP
SIL127SHEXNULL:
        ret


; IHEX =========================================================================
; Parameters:
;       SOURCE - r9
;       TARGET - r8
; Converts integer or real into hex digits
; Input:
;         ------------------------------
; SOURCE  |   I    |         |         |
;         ------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   CL16  |
;         ----------------------------------------
; where Cn are the hex digits representing the integer
; The hex digits are in big endian order.. leftmost is high order, rightmost is
;  low order
; L2 should equal 16 (ALENG*BITSPA/4)
SIL127IHEX:
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        mov     rax,[r9]                ;Get integer or real
        mov     rcx,ALENG*BITSPA/4      ;number of hex digits
SIL127IHEXLP:
        rol     rax,4                   ;look at high order part first
        mov     bl,al
        and     rbx,15
        cmp     rbx,10
        jl      SIL127IHEXd
        add     bl,'A'-10
        jmp     SIL127IHEXg
SIL127IHEXd:
        or      bl,'0'
SIL127IHEXg:
        mov     byte ptr [rdi],bl
        inc     rdi                     ;next output position
        loop    SIL127IHEXLP
        ret

; R2HS =========================================================================
; Parameters:
;       SOURCE - r9
;       TARGET - r8
; Converts real to 8 hex digits of single precision
; Input:
;         ------------------------------
; SOURCE  |   I    |         |         |
;         ------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   CL8   |
;         ----------------------------------------
; where Cn are the hex digits representing the integer
; The hex digits are in big endian order.. leftmost is high order, rightmost is
;  low order
; L2 should equal 8
SIL127R2HS:
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        fld     qword ptr [r9]          ;Get real
        fstp    dword ptr WD0           ;Save as single precision
        mov     eax,dword ptr WD0
        mov     rcx,8                   ;number of hex digits
        rol     rax,32                  ;move it to high order half
SIL127R2HSLP:
        rol     rax,4                   ;look at high order part first
        mov     bl,al
        and     bl,15
        add     bl,'0'
        cmp     bl,'9'
        jle     @F
        add     bl,'A'-'0'-10
@@:
        mov     byte ptr [rdi],bl       ;put hex digit in result area
        inc     rdi                     ;next output position
        loop    SIL127R2HSLP
        ret

; R2HX =========================================================================
; Parameters:
;       SOURCE - r9
;       TARGET - r8
; Converts real to 20 hex digits of extended double precision
; Input:
;         ------------------------------
; SOURCE  |   I    |         |         |
;         ------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   CL20  |
;         ----------------------------------------
; where Cn are the hex digits representing the integer
; The hex digits are in big endian order.. leftmost is high order, rightmost is
;  low order
; L2 should equal 8
SIL127R2HX:
        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET]        ;target address
        fld     qword ptr [r9]          ;Get real
        wait
        fstp    tbyte ptr WD0           ;Save as extended double precision
        wait
        lea     r8,wd0+9
        mov     rcx,10                  ;number of bytes of input
SIL127R2HXLP:
        mov     al,[r8]
        rol     rax,60                  ;look at high order part first
        mov     bl,al
        and     bl,15
        add     bl,'0'
        cmp     bl,'9'
        jle     @F
        add     bl,'A'-'0'-10
@@:     mov     byte ptr [rdi],bl       ;put hex digit in result area
        inc     rdi
        rol     rax,4                   ;look at low order part of byte
        mov     bl,al
        and     bl,15
        add     bl,'0'
        cmp     bl,'9'
        jle     @F
        add     bl,'A'-'0'-10
@@:     mov     byte ptr [rdi],bl       ;put hex digit in result area
        inc     rdi                     ;next output position
        dec     r8                      ;prior byte in tbyte area
        loop    SIL127R2HXLP
        ret

; B2S ==========================================================================
; Converts bit string to byte string
; Parameters:
; r9:   SOURCE
; r8:   TARGET
; Input:
;         ---\-----------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L2) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
; where Sn are the bits representing the string S (source operand)
; The bits are not in any endian order.. leftmost is high order, rightmost is low order
; L2 should equal (L1+BITSPA-1)/BITSPA  source bits are padded on right to get byte multiple
SIL127B2S:
        mov     rsi,[r9]
        add     rsi,[r9+OOFFSET] ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8            ;source length

        mov     rdi,[r8]
        add     rdi,[r8+OOFFSET] ;target address

        jecxz   SIL127B2SNULL
SIL127B2SBYTELP:
        mov     rdx,BITSPA      ;bits per byte
        mov     AH,0            ;zero new output byte
SIL127B2SBITLP:
        mov     al,[rsi]        ;Get string byte
        rcr     al,1            ;look at high order bit first
        rcl     ah,1            ;build up output byte
        inc     rsi
        dec     rcx
        jnz     SIL127B2SMS     ;jump if there is more source
        cmp     rdx,1
        jz      SIL127B2SMS     ;see if no pads will be needed
        mov     rcx,1
        lea     rsi,ZEROINT     ;fake zero pad bits
SIL127B2SMS:                    ;more stuff
        dec     rdx
        jg      SIL127B2SBITLP
        mov     [rdi],ah
        mov     ah,0
        inc     rdi
        cmp     rcx,0
        jnz     SIL127B2SBYTELP ;do all source bytes
SIL127B2SNULL:
        ret



; H2B ==========================================================================
; Parameters:
; r9    SOURCE - 4 byte specifier of source
; r8    TARGET - 4 byte specifier for target
; Input:
;         ---\-----------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L2) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
; where Sn are the hex digits (source)
; where Cn are the bit digits (target)
; The leftmost hex digit is high order, rightmost is low order
; L2 will always be 4 times L1.
SILH2BFUN:
        mov     rdi,[r8+AOFFSET] ;Target specifier address ptr
        add     rdi,[r8+OOFFSET] ;String address
        mov     rsi,[r9+AOFFSET] ;Source specifier address ptr
        add     rsi,[r9+OOFFSET] ;String address
        mov     rcx,[r9+LOFFSET] ;Source string length
        shr     rcx,8            ;source length corrected
        jrcxz   SILH2BFUNNULL
        mov     dl,4            ;bits per hex digit
        xor     rbx,rbx
        xor     rax,rax
        lea     r8,HEXTABLE
SILH2BFUNLP:
        mov     bl,[rsi]
        mov     al,[r8+rbx] ;convert hex digit to binary
        mov     dl,'0'
        test    al,8
        jz      @F
        mov     dl,'1'
@@:     mov     [rdi],dl
        inc     rdi

        mov     dl,'0'
        test    al,4
        jz      @F
        mov     dl,'1'
@@:     mov     [rdi],dl
        inc     rdi

        mov     dl,'0'
        test    al,2
        jz      @F
        mov     dl,'1'
@@:     mov     [rdi],dl
        inc     rdi

        mov     dl,'0'
        test    al,1
        jz      @F
        mov     dl,'1'
@@:     mov     [rdi],dl
        inc     rdi

        inc     rsi
        loop    SILH2BFUNLP

SILH2BFUNNULL:
        ret


; R2BY ======================================================================
; Parameters:
; r9    SOURCE - specifier address of source
; r8    STRING - descriptor address of target
; Input:
;         ------------------------------
; SOURCE  |   n    |    0    |  I or R |
;         ------------------------------
; Output:
;         --------------------------------------------------
; TARGET  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   L1    |
;         ----------------------------------------
; where Cn are the bytes in little endian that represent the integer or real
; Leftmost byte is low order, rightmost is high order
; L1 should equal 8 (ALENG) for integer version
; The Integer and Real versions work almost the same
SIL127I2BY:                             ;version for i2bytes(s)
SIL127R2BY:                             ;version for r2bytes(s)
        mov     rdi,[r8+OOFFSET]
        add     rdi,[r8]                ;target address
;       mov     rcx,[r8+LOFFSET]        ;length should have been preset
;       shr     rcx,8                   ;length of target should be 8
        mov     rsi,r9
        mov     rcx,8                   ;length 8 bytes
        rep movsb                       ;move to target
        ret

SIL127R2BYS:                            ;version for r2bytess(s)
        mov     rdi,[r8+OOFFSET]
        add     rdi,[r8]                ;target address
;       mov     rcx,[r8+LOFFSET]        ;lenght should have been preset
;       shr     rcx,8                   ;length of target should be 4
        fld     qword ptr [r9]          ;get double precision value
        fwait
        fstp    dword ptr [rdi]         ;store as single precision value
        ret

SIL127R2BYX:                            ;version for r2bytesx(s)
        mov     rdi,[r8+OOFFSET]
        add     rdi,[r8]                ;target address
;       mov     rcx,[r8+LOFFSET]        ;lenght should have been preset
;       shr     rcx,8                   ;length of target should be 10
        fld     qword ptr [r9]          ;get double precision value
        fwait
        fstp    tbyte ptr [rdi]         ;store as extended precision value
        ret

; BY2I ======================================================================
; Parameters:
; r9    SOURCE - specifier address of source
; r8    STRING - descriptor address of target
; Input:
;         --------------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   L1    |
;         ----------------------------------------
; Output:
;         ------------------------------
; TARGET  |   n    |    0    |  I or R |
;         ------------------------------
; where Cn are the bytes in little endian that represent the integer
; Leftmost byte is low order, rightmost is high order
; L1 should equal 8 (ALENG) or less for integer version
; The Integer and Real versions work almost the same
SIL127BY2I:                             ;version for bytes2i(s)
        mov     r10,I*256               ;to set F and V fields of result
        jmp     SIL127BY2II             ;Integer case

SIL127BY2IS:                            ;version for bytes2is(s) (sign extended)
        mov     qword ptr [r8+VOFFSET],I*256    ;set F and V fields of result
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        mov     r11,8                   ;max length
        xor     rax,rax
        mov     qword ptr [r8+AOFFSET],rax ;clear to all zero
        jrcxz   SIL127BY2ISNL           ;jump for null string
        cmp     rcx,ALENG               ;limit length
        jna     SIL127BY2ISLP
        mov     rcx,ALENG
SIL127BY2ISLP:
        mov     bl,[rsi]        ;Get byte
        mov     byte ptr [r8+AOFFSET],bl
        inc     rsi             ;next source byte address
        inc     r8              ;next target address
        dec     r11             ;max left to do
        loop    SIL127BY2ISLP
        cmp     bl,127
        jna     SIL127BY2ISNL           ;no sign to extend
        mov     rcx,r11                 ;this many more bytes of ff needed
        jrcxz   SIL127BY2ISNL           ;jump if got all 8 bytes
SIL127BY2ISE:
        mov     byte ptr [r8+AOFFSET],255
        inc     r8
        loop    SIL127BY2ISE
SIL127BY2ISNL:
        ret

SIL127BY2R:                             ;for bytes2r(s) (double precision)
        mov     r10,R*256               ;to set F and V fields of result
        jmp     SIL127BY2II             ;Same code as for integer

SIL127BYS2R:                            ;for bytess2r(s) (single precision)
        mov     qword ptr [r8+VOFFSET],R*256 ;set F and V fields of result
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
;       mov     rcx,[r9+LOFFSET]        ;length should have been already checked
;       shr     rcx,8                   ;length of source, should be 4 bytes
        fld     dword ptr [rsi]         ;load single precision value
        fwait
        fstp    qword ptr [r8+AOFFSET]  ;store as double precision
        ret

SIL127BYX2R:                            ;for bytess2r(s) (single precision)
        mov     qword ptr [r8+VOFFSET],R*256 ;set F and V fields of result
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
;       mov     rcx,[r9+LOFFSET]        ;length should have been already checked
;       shr     rcx,8                   ;length of source, should be 10 bytes
        fld     tbyte ptr [rsi]         ;load extended precision value
        fwait
        fstp    qword ptr [r8+AOFFSET]  ;store as double precision
        ret

SIL127BY2II:                            ;integer case
        mov     qword ptr [r8+VOFFSET],r10      ;set F and V fields of result
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        xor     rax,rax
        mov     qword ptr [r8+AOFFSET],rax ;clear to all zero
        jrcxz   SIL127BY2IINL           ;jump for null string
        cmp     rcx,ALENG               ;limit length
        jna     SIL127BY2IILP
        mov     rcx,ALENG
SIL127BY2IILP:
        mov     bl,[rsi]        ;Get byte
        mov     byte ptr [r8+AOFFSET],bl
        inc     rsi             ;next source byte address
        inc     r8              ;next target address
        loop    SIL127BY2IILP
SIL127BY2IINL:
        ret


; H2I H2R HS2R HX2R =========================================================
; Parameters:
; r9    SOURCE - 4 byte specifier address of source
; r8    STRING - 4 byte descriptor address of target
; Input:
;         --------------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   CL64  |
;         ----------------------------------------
; Output:
;         ------------------------------
; TARGET  |   n    |    0    |  I or R |
;         ------------------------------
; where Cn are the bytes of hexadecimal characters that represent the integer
; Leftmost hex digit is high order, rightmost is low order
; L2 should equal 16 (ALENG*2) or less
; The Integer and Real versions work almost the same
SIL127H2I:
        mov     r10,I*256               ;to set F and V fields of result
        jmp     SIL127H2RI              ;Integer case

SIL127HS2R:
        mov     r10,R*256               ;to set F and V fields of result
        jmp     SIL127HS2RX             ;Single precision case

SIL127HX2R:
        mov     r10,R*256               ;to set F and V fields of result
        jmp     SIL127HX2RX             ;Double extended precision case

SIL127H2R:                              ;Double precision case
        mov     r10,R*256               ;to set F and V fields of result

SIL127H2RI:                             ;double real and integer case
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        lea     r9,HEXTABLE
        xor     rax,rax
        xor     rbx,rbx
        jrcxz   SIL127H2INL
SIL127H2ILP:
        mov     bl,[rsi]        ;Get byte
        shl     rax,4
        or      al,[r9+rbx]     ;get hex value
        inc     rsi             ;next byte address
        loop    SIL127H2ILP
SIL127H2INL:
        mov     [r8],rax        ;result value
        mov     qword ptr [r8+VOFFSET],r10      ;set F and V fields of result
        ret

SIL127HX2RX:                            ;double extended precision case
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        lea     r9,HEXTABLE
        xor     rax,rax
        xor     rbx,rbx
        lea     rdi,sil127rxwk+9        ;place to assemble extended float
        jrcxz   SIL127H2INLX            ;should not happen, checked for 20 digits
SIL127H2ILPX:
        mov     bl,[rsi]        ;Get byte
        shl     rax,4
        or      al,[r9+rbx]     ;get hex value
        inc     rsi             ;next byte address
        dec     rcx
        mov     bl,[rsi]        ;Get byte
        shl     rax,4
        or      al,[r9+rbx]     ;get hex value
        mov     byte ptr [rdi],al
        dec     rdi
        inc     rsi             ;next byte address
        loop    SIL127H2ILPX
SIL127H2INLX:
        fld     TBYTE PTR sil127rxwk    ;load ten byte value
        wait
        fstp    QWORD PTR [r8]  ;result value in double precision
        wait
        mov     qword ptr [r8+VOFFSET],r10      ;set F and V fields of result
        ret

SIL127HS2RX:    ;Single precision case
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        lea     r9,HEXTABLE
        xor     rax,rax
        xor     rbx,rbx
        lea     rdi,sil127rxwk+3        ;place to assemble extended float
        jrcxz   SIL127H2INLS            ;should not happen, checked for 8 digits
SIL127H2ILPS:
        mov     bl,[rsi]        ;Get byte
        shl     rax,4
        or      al,[r9+rbx]     ;get hex value
        inc     rsi             ;next byte address
        dec     rcx
        mov     bl,[rsi]        ;Get byte
        shl     rax,4
        or      al,[r9+rbx]     ;get hex value
        mov     byte ptr [rdi],al
        dec     rdi
        inc     rsi             ;next byte address
        loop    SIL127H2ILPS
SIL127H2INLS:
        fld     DWORD PTR sil127rxwk    ;load 4 byte value
        wait
        fstp    QWORD PTR [r8]  ;result value in double precision
        wait
        mov     qword ptr [r8+VOFFSET],r10      ;set F and V fields of result
        ret

; h2is ==========================================================================
; Parameters:
; r9    SOURCE - 4 byte specifier address of source
; r8    STRING - 4 byte descriptor address of target
; Input:
;         --------------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   CL64  |
;         ----------------------------------------
; Output:
;         ------------------------------
; TARGET  |   n    |    0    |  I or R |
;         ------------------------------
; where Cn are the bytes of hexadecimal characters that represent the integer
; Leftmost hex digit is high order, rightmost is low order
; Left most digit is sign extened if there are few than ALENG*2 digits
; L2 should equal 16 (ALENG*2) or less
; The Integer and Real versions work almost the same
SIL127H2IS:
        mov     r10,I*256               ;to set F and V fields of result
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        lea     r9,HEXTABLE
        xor     rax,rax
        xor     rbx,rbx
        jrcxz   SIL127h2isNL

        mov     bl,[rsi]        ;Get leftmost hex digit
        mov     bl,[r9+rbx]     ;get hex value
        test    bl,8            ;check high order bit
        jz      SIL127h2isLP
        dec     rax             ;set all bits on (to sign extend)
SIL127h2isLP:
        mov     bl,[rsi]        ;Get byte
        shl     rax,4
        or      al,[r9+rbx]     ;get hex value
        inc     rsi             ;next byte address
        loop    SIL127h2isLP
SIL127h2isNL:
        mov     [r8],rax        ;result value
        mov     qword ptr [r8+VOFFSET],r10      ;set F and V fields of result
        ret


; H2S ==========================================================================
; Parameters:
; r9    SOURCE - 4 byte specifier of source
; r8    TARGET - 4 byte specifier for target
; Input:
;         ---\-----------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L2) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
; where Sn are the hex digits (source)
; where Cn are the characters (target)
; The leftmost hex digit is high order, rightmost is low order
; If L1 is odd, then there is an implied zero hex digit appended
; L2 will always be (L1+1)/2.
SILH2SFUN:
        mov     rdi,[r8+AOFFSET] ;Target specifier address ptr
        add     rdi,[r8+OOFFSET] ;String address
        mov     rsi,[r9+AOFFSET] ;Source specifier address ptr
        add     rsi,[r9+OOFFSET] ;String address
        mov     rcx,[r9+LOFFSET] ;Source string length
        shr     rcx,8            ;source length corrected
        jrcxz   SILH2SFUNNULL
        mov     dl,4            ;bits per hex digit
        xor     rbx,rbx
        xor     rax,rax
        lea     r8,HEXTABLE
SILH2SFUNLP:
        mov     bl,[rsi]
        mov     dl,[r8+rbx] ;convert hex digit to binary
        shl     dl,4

        inc     rsi
        dec     rcx
        jnz     @F
        mov     [rdi],dl        ;put last result char (right zero)
        ret
@@:     mov     bl,[rsi]
        or      dl,[r8+rbx] ;convert hex digit to binary
        mov     [rdi],dl        ;put result char
        inc     rdi
        inc     rsi
        loop    SILH2SFUNLP
SILH2SFUNNULL:
        ret



; B2H ==========================================================================
; Parameters:
; r9    SOURCE - 4 byte specifier of source
; r8    TARGET - 4 byte specifier for target
; Input:
;         ---\-----------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         --------------------------------------------------
; TARGET  |   A2   |         |         |    O2   |    L2   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   S1   |   S2    |  ...    |   S(L2) |
;         ----------------------------------------
; Output:
;         ----------------------------------------
; A2+O2   |   C1   |   C2    |  ...    |   C(L2) |
;         ----------------------------------------
; where Sn are the bits representing the string S (source operand)
; The bits are not in any endian order.. leftmost is high order, rightmost is low order
; L2 should equal (L1+BITSPA-1)/BITSPA  source bits are padded on right to get hex digit multiple
SIL127B2H:
        mov     rdi,[r8+AOFFSET] ;Target specifier address ptr
        add     rdi,[r8+OOFFSET] ;String address
        mov     rsi,[r9+AOFFSET] ;Source specifier address ptr
        add     rsi,[r9+OOFFSET] ;String address
        mov     rcx,[r9+LOFFSET] ;Source string length
        shr     rcx,8            ;target length corrected
        JECXZ   SIL127B2HNULL
SIL127B2HBYTELP:
        MOV     DL,4            ;bits per hex digit
        MOV     Al,0            ;zero new output byte
SIL127B2HBITLP:
        RCL     al,1            ;build up output byte
        MOV     bl,[rsi]        ;Get string byte
        AND     bl,1            ;look at only low order bit
        OR      al,bl           ;insert the bit
        INC     rsi
        dec     rcx
        JNZ     SIL127B2HMS     ;jump if there is more source
        CMP     DL,1
        JZ      SIL127B2HMS     ;see if no pads will be needed
        MOV     RCX,1
        MOV     RSI,Offset ZEROINT ;fake zero pad bits
SIL127B2HMS:                    ;more stuff
        DEC     DL
        JNZ     SIL127B2HBITLP
        ADD     AL,'0'
        CMP     AL,'9'
        JNA     SIL127B2Hnotlet
        ADD     AL,'A'-'9'-1
SIL127B2Hnotlet:
        MOV     [RDI],AL
        INC     RDI
        CMP     RCX,0
        JNZ     SIL127B2HBYTELP ;do all source bytes
SIL127B2HNULL:
        ret


; B2I ==========================================================================
; Parameters:
; r9    SOURCE - 4 byte specifier address of source
; r8    STRING - 4 byte descriptor address of target
; Input:
;         --------------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   CL64  |
;         ----------------------------------------
; Output:
;         ------------------------------
; TARGET  |   n    |    0    |  I or R |
;         ------------------------------
; where Cn are the bytes, of which the low order bits represent the integer
; The bits are not in any endian order.. leftmost is high order, rightmost is low order
; L2 should equal 64 (ALENG*BITSPA) or less
; The Integer and Real versions work almost the same
SIL127B2I:
        mov     r10,I*256               ;to set F and V fields of result
        jmp     SIL127B2RI
SIL127B2R:
        mov     r10,R*256               ;to set F and V fields of result
SIL127B2RI:
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        xor     rbx,rbx
        CMP     rcx,0
        JZ      SIL127B2INL
SIL127B2ILP:
        mov     al,[rsi]        ;Get byte
        ror     al,1            ;move bit into carry
        rcl     rbx,1
        inc     rsi             ;next byte address
        loop    SIL127B2ILP
SIL127B2INL:
        mov     [r8],rbx        ;result value
        mov     qword ptr [r8+VOFFSET],r10      ;set F and V fields of result
        ret

; B2IS ==========================================================================
; Parameters:
; r9    SOURCE - 4 byte specifier address of source
; r8    STRING - 4 byte descriptor address of target
; Input:
;         --------------------------------------------------
; SOURCE  |   A1   |         |         |    O1   |    L1   |
;         --------------------------------------------------
;         ----------------------------------------
; A1+O1   |   C1   |   C2    |  ...    |   CL64  |
;         ----------------------------------------
; Output:
;         ------------------------------
; TARGET  |   n    |    0    |  I or R |
;         ------------------------------
; where Cn are the bytes, of which the low order bits represent the integer
; The bits are not in any endian order.. leftmost is high order, rightmost is low order
; L2 should equal 64 (ALENG*BITSPA) or less
; The Integer is sign bit exteneded on the left
SIL127B2IS:
        mov     rsi,[r9+OOFFSET]
        add     rsi,[r9]                ;source address
        mov     rcx,[r9+LOFFSET]
        shr     rcx,8                   ;length of source
        add     rsi,rcx                 ;last+1 character of source
        xor     rbx,rbx                 ;result will go here
        mov     rdx,ALENG*BITSPA        ;will need this many bits
        CMP     rcx,0
        JZ      SIL127B2ISNL            ;zero length input produces zero
SIL127B2ISLP:
        dec     rsi                     ;next byte address
        mov     al,[rsi]                ;Get byte
        ror     al,1                    ;move bit into carry
        rcr     rbx,1                   ;move bit into result
        dec     rdx                     ;one more bit done
        loop    SIL127B2ISLP            ;byte loop
        and     rdx,rdx
        jz      SIL127B2ISNL            ;did we do 64 bits?
        mov     rcx,rdx                 ;this many bits to extend
SIL127B2ISBLP:
        mov     al,[rsi]                ;Get byte
        ror     al,1                    ;move bit into carry
        rcr     rbx,1                   ;move bit into result
        loop    SIL127B2ISBLP           ;one more bit done
SIL127B2ISNL:
        mov     [r8],rbx                ;result value
        mov     qword ptr [r8+VOFFSET],I*256 ;set F and V fields of result
        ret

; SYSTEM =======================================================================
; Parameters:
; rcx   length of command
; rsi   text of command
systcmd proc
        ;#### make sure rcx is not too big 32768-8
ifdef linuxenvironment
        lea     rdi,systcommand
        rep movsb
        mov     byte ptr [rdi],0
        mov     rax,58          ;sys_vfork
        syscall
        cmp     eax,-1 ;error
        jz      notnewprocfail
        cmp     eax,0
        jnz     notnewproc

        mov     rax,59          ;sys_execve
        lea     rdi,lnxcmdname
        lea     rsi,lnxargsp
        mov     rdx,linuxenvironmentvars ;envbuffer
        syscall                 ;see if this runs the command
        dbgustr 'sys_execve failed'
        mov     rax,60          ;sys_exit
        mov     rdi,0
        syscall                 ;only gets here if sys_execcve failed

notnewprocfail:
        dbgustr 'sys_vfork failed'
notnewproc:
        mov     rdi,rax         ;pid of forked process
        xor     rsi,rsi         ;null
        xor     rdx,rdx         ;options
        xor     r10,r10         ;rusage
        mov     rax,61          ;sys_wait4
        syscall

else                            ;Windows version ==============================
        push    rsi
        push    rcx
        lea     rdi,systcommand
        mov     rcx,7
        lea     rsi,wincmdname
        rep movsb
        pop     rcx
        pop     rsi
        rep movsb
        mov     byte ptr [rdi],0

; #### trying new version
        externdef CreateProcessA : near
        winstack 6
        xor     rcx,rcx         ; application name
        lea     rdx,systcommand ; command line
        xor     r8,r8           ; process attributes
        xor     r9,r9           ; thread attributes
        mov     qword ptr [rsp+32],0 ; inherit handles
NORMAL_PRIORITY_CLASS equ 20h
        mov     qword ptr [rsp+40],NORMAL_PRIORITY_CLASS ; creation flgs
        mov     qword ptr [rsp+48],0 ; Environment
        mov     qword ptr [rsp+56],0 ; Current Directory
        lea     rax,winjunk1 ;####
        mov     qword ptr [rsp+64],rax ; Starup info
        lea     rax,winjunk2 ;####
        mov     qword ptr [rsp+72],rax ; Process information
        call    CreateProcessA
        winstacke
        cmp     rax,0
        jnz     cpnz
        dbgustr 'Error from CreateProcessA'
        call    dbggle
cpnz:

        externdef WaitForSingleObject : near
        winstack
        mov     rcx,winjunk2    ;handle is first thing here
        xor     rdx,rdx
        dec     rdx
        call    WaitForSingleObject
        winstacke
        jnz     cpnz2
        dbgustr 'Error from WaitForSingleObject'
        call    dbggle
cpnz2:

endif
;dbgustr 'at end of system command ####'
;call dumpunit ;####
        ret
systcmd endp

; SLEEP ========================================================================
; Sleep for RAX nanoseconds
; Parameters:
;    t:    Sleep value in RAX
; Input:
;         ------------------------------
; SOURCE1 |   t    |         |         |
;         ------------------------------
SILSLEEP:
        mov     silsleeping,1   ;say we are sleeping so ctrl-c works
ifdef linuxenvironment          ;LINUX version ===============================
        xor     rdx,rdx
        div     silbillion
        cmp     rax,silbillion
        jl      @F
        mov     rax,silbillion     ;limit it to billion seconds
@@:     mov     siltimespecn,edx ;nanosecond part
        mov     siltimespecs,eax ;seconds part
NANOSLEEP equ   35
        mov     rax,NANOSLEEP
        lea     rdi,siltimespecs
        lea     rsi,siltimespecs
        syscall
else                            ;Windows version =============================
        externdef SleepEx : near
        externdef Sleep : near
        xor     rdx,rdx
        div     silmillion
        mov     rcx,rax         ;milliseconds value
        xor     rdx,rdx
        dec     rdx             ;true
        winstack
        call    Sleep
        winstacke
endif
        mov     silsleeping,0
        ret

; ANDFUN =======================================================================
; Performs logical AND function
; Parameters:
; r8:   TARGET
; r9:   SOURCE1
; r10:  SOURCE2
; Input:
;         ----------------------------------------------------
; SOURCE1 |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; SOURCE2 |   A2   |         |         |    O2    |    L2    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A3   |         |         |    O3    |MAX(L1,L2)|
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
;         -----------------------------------------
; A2+O2   |   Y1   |   Y2    |  ...    |   Y(L2)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A3+O3   |   Z1   |   Z2    |  ...    |MAX(L1,L2)|
;         -----------------------------------------
SILANDFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     r9,[r10+AOFFSET] ;source2 address
        add     r9,[r10+OOFFSET] ;source2 offset
        mov     rdx,[r10+LOFFSET]
        shr     rdx,8            ;source2 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILANDFUNlp1:
        cmp     rbx,r10
        jl      SILANDFUNs      ;jump if short amount left
        cmp     rdx,r10
        jl      SILANDFUNs      ;jump if short amount left

        mov     rax,[rsi]
        and     rax,[r9]
        mov     [rdi],rax
        add     rsi,r10
        add     r9,r10
        add     rdi,r10
        sub     rbx,r10
        sub     rdx,r10
        sub     rcx,r10
        jmp     SILANDFUNlp1
SILANDFUNs:
        and     rbx,rbx
        jz      SILANDFUNf      ;jump if one has run out
        and     rdx,rdx
        jz      SILANDFUNf      ;jump if one has run out
        mov     al,[rsi]
        and     al,[r9]
        mov     [rdi],al
        inc     rsi
        inc     r9
        inc     rdi
        dec     rbx
        dec     rdx
        dec     rcx
        jmp     SILANDFUNs
SILANDFUNf:
        jrcxz   SILANDFUNdone
        mov     rax,0
SILANDFUNlp:
        rep stosb
SILANDFUNdone:
        ret

; NANDFUN =====================================================================
; Performs logical NAND function
; Parameters:
; r8:   TARGET
; r9:   SOURCE1
; r10:  SOURCE2
; Input:
;         ----------------------------------------------------
; SOURCE1 |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; SOURCE2 |   A2   |         |         |    O2    |    L2    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A3   |         |         |    O3    |MAX(L1,L2)|
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
;         -----------------------------------------
; A2+O2   |   Y1   |   Y2    |  ...    |   Y(L2)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A3+O3   |   Z1   |   Z2    |  ...    |MAX(L1,L2)|
;         -----------------------------------------
SILNANDFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     r9,[r10+AOFFSET] ;source2 address
        add     r9,[r10+OOFFSET] ;source2 offset
        mov     rdx,[r10+LOFFSET]
        shr     rdx,8            ;source2 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILNANDFUNlp1:
        cmp     rbx,r10
        jl      SILNANDFUNs      ;jump if short amount left
        cmp     rdx,r10
        jl      SILNANDFUNs      ;jump if short amount left

        mov     rax,[rsi]
        and     rax,[r9]
        not     rax
        mov     [rdi],rax
        add     rsi,r10
        add     r9,r10
        add     rdi,r10
        sub     rbx,r10
        sub     rdx,r10
        sub     rcx,r10
        jmp     SILNANDFUNlp1
SILNANDFUNs:
        and     rbx,rbx
        jz      SILNANDFUNf      ;jump if one has run out
        and     rdx,rdx
        jz      SILNANDFUNf      ;jump if one has run out
        mov     al,[rsi]
        and     al,[r9]
        not     al
        mov     [rdi],al
        inc     rsi
        inc     r9
        inc     rdi
        dec     rbx
        dec     rdx
        dec     rcx
        jmp     SILNANDFUNs
SILNANDFUNf:
        jrcxz   SILNANDFUNdone
        mov     al,0FFh
SILNANDFUNlp:
        rep stosb
SILNANDFUNdone:
        ret

; NORFUN =======================================================================
; Performs logical NOR function
; Parameters:
; r8:   TARGET
; r9:   SOURCE1
; r10:  SOURCE2
; Input:
;         ----------------------------------------------------
; SOURCE1 |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; SOURCE2 |   A2   |         |         |    O2    |    L2    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A3   |         |         |    O3    |MAX(L1,L2)|
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
;         -----------------------------------------
; A2+O2   |   Y1   |   Y2    |  ...    |   Y(L2)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A3+O3   |   Z1   |   Z2    |  ...    |MAX(L1,L2)|
;         -----------------------------------------
; Zn = Xn NOR Yn
SILNORFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     r9,[r10+AOFFSET] ;source2 address
        add     r9,[r10+OOFFSET] ;source2 offset
        mov     rdx,[r10+LOFFSET]
        shr     rdx,8            ;source2 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILNORFUNlp1:
        cmp     rbx,r10
        jl      SILNORFUNs      ;jump if short amount left
        cmp     rdx,r10
        jl      SILNORFUNs      ;jump if short amount left

        mov     rax,[rsi]
        OR      rax,[r9]
        not     rax
        mov     [rdi],rax
        add     rsi,r10
        add     r9,r10
        add     rdi,r10
        sub     rbx,r10
        sub     rdx,r10
        sub     rcx,r10
        jmp     SILNORFUNlp1
SILNORFUNs:
        and     rbx,rbx
        jz      SILNORFUNfb     ;jump if one has run out
        and     rdx,rdx
        jz      SILNORFUNfd     ;jump if one has run out
        mov     al,[rsi]
        OR      al,[r9]
        not     al
        mov     [rdi],al
        inc     rsi
        inc     r9
        inc     rdi
        dec     rbx
        dec     rdx
        dec     rcx
        jmp     SILNORFUNs
SILNORFUNfb:
        jrcxz   SILNORFUNdone
        mov     rcx,rdx
        mov     rsi,r9
        rep movsb

SILNORFUNfd:
        jrcxz   SILNORFUNdone
        mov     rcx,rbx
SILNORFUNlp2:
        mov     al,[rsi]
        not     al
        mov     [rdi],al
        inc     rdi
        inc     rsi
        loop    SILNORFUNlp2

SILNORFUNdone:
        ret


; LOBFUN =======================================================================
; Makes low order bit into 0 or 1 character
; Parameters:
; r8:   TARGET
; r9:   SOURCE
; Input:
;         ----------------------------------------------------
; SOURCE  |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A2   |         |         |    O2    |    L1    |
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A2+O2   |   Z1   |   Z2    |  ...    |   Z(L1)  |
;         -----------------------------------------
; If the low order bit of Xn is zero, then Zn is set to the character "0".
; If the low order bit of Xn is one, then Zn is set to the character "1".
SILLOBFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILLOBFUNlp1:
        cmp     rcx,r10
        jl      SILLOBFUNs      ;jump if short amount left

        mov     rax,[rsi]
        and     rax,LOWBITS8
        or      rax,CHAR8ONES
        mov     [rdi],rax
        add     rsi,r10
        add     rdi,r10
        sub     rcx,r10
        jmp     SILLOBFUNlp1
SILLOBFUNs:
        jrcxz   SILLOBFUNdone   ;jump if has run out
SILLOBFUNss:
        mov     al,[rsi]
        and     al,1
        or      al,'0'
        mov     [rdi],al
        inc     rsi
        inc     rdi
        loop    SILLOBFUNss

SILLOBFUNdone:
        ret


; NOTFUN =======================================================================
; Performs logical NOT function
; Parameters:
; r8:   TARGET
; r9:   SOURCE
; Input:
;         ----------------------------------------------------
; SOURCE  |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A2   |         |         |    O2    |    L1    |
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A2+O2   |   Z1   |   Z2    |  ...    |   Z(L1)  |
;         -----------------------------------------
; Zn = Xn NOT Yn
SILNOTFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILNOTFUNlp1:
        cmp     rcx,r10
        jl      SILNOTFUNs      ;jump if short amount left

        mov     rax,[rsi]
        not     rax
        mov     [rdi],rax
        add     rsi,r10
        add     rdi,r10
        sub     rcx,r10
        jmp     SILNOTFUNlp1
SILNOTFUNs:
        jrcxz   SILNOTFUNdone   ;jump if has run out
SILNOTFUNss:
        mov     al,[rsi]
        not     al
        mov     [rdi],al
        inc     rsi
        inc     rdi
        loop    SILNOTFUNss

SILNOTFUNdone:
        ret


; ORFUN =======================================================================
; Performs logical OR function
; Parameters:
; r8:   TARGET
; r9:   SOURCE1
; r10:  SOURCE2
; Input:
;         ----------------------------------------------------
; SOURCE1 |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; SOURCE2 |   A2   |         |         |    O2    |    L2    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A3   |         |         |    O3    |MAX(L1,L2)|
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
;         -----------------------------------------
; A2+O2   |   Y1   |   Y2    |  ...    |   Y(L2)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A3+O3   |   Z1   |   Z2    |  ...    |MAX(L1,L2)|
;         -----------------------------------------
; Zn = Xn OR Yn
SILORFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     r9,[r10+AOFFSET] ;source2 address
        add     r9,[r10+OOFFSET] ;source2 offset
        mov     rdx,[r10+LOFFSET]
        shr     rdx,8            ;source2 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILORFUNlp1:
        cmp     rbx,r10
        jl      SILORFUNs      ;jump if short amount left
        cmp     rdx,r10
        jl      SILORFUNs      ;jump if short amount left

        mov     rax,[rsi]
        OR      rax,[r9]
        mov     [rdi],rax
        add     rsi,r10
        add     r9,r10
        add     rdi,r10
        sub     rbx,r10
        sub     rdx,r10
        sub     rcx,r10
        jmp     SILORFUNlp1
SILORFUNs:
        and     rbx,rbx
        jz      SILORFUNfb     ;jump if one has run out
        and     rdx,rdx
        jz      SILORFUNfd     ;jump if one has run out
        mov     al,[rsi]
        OR      al,[r9]
        mov     [rdi],al
        inc     rsi
        inc     r9
        inc     rdi
        dec     rbx
        dec     rdx
        dec     rcx
        jmp     SILORFUNs
SILORFUNfb:
        jrcxz   SILORFUNdone
        mov     rcx,rdx
        mov     rsi,r9
        rep movsb

SILORFUNfd:
        jrcxz   SILORFUNdone
        mov     rcx,rbx
        rep movsb

SILORFUNdone:
        ret


; XORFUN =======================================================================
; Performs logical eXclusive OR function
; Parameters:
; r8:   TARGET
; r9:   SOURCE1
; r10:  SOURCE2
; Input:
;         ----------------------------------------------------
; SOURCE1 |   A1   |         |         |    O1    |    L1    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; SOURCE2 |   A2   |         |         |    O2    |    L2    |
;         ----------------------------------------------------
;         ----------------------------------------------------
; TARGET  |   A3   |         |         |    O3    |MAX(L1,L2)|
;         ----------------------------------------------------
;         -----------------------------------------
; A1+O1   |   X1   |   X2    |  ...    |   X(L1)  |
;         -----------------------------------------
;         -----------------------------------------
; A2+O2   |   Y1   |   Y2    |  ...    |   Y(L2)  |
;         -----------------------------------------
; Output:
;         -----------------------------------------
; A3+O3   |   Z1   |   Z2    |  ...    |MAX(L1,L2)|
;         -----------------------------------------
; Zn = Xn XOR Yn
SILXORFUN:
        mov     rsi,[r9+AOFFSET] ;source1 address
        add     rsi,[r9+OOFFSET] ;source1 offset
        mov     rbx,[r9+LOFFSET]
        shr     rbx,8            ;source1 length
        mov     r9,[r10+AOFFSET] ;source2 address
        add     r9,[r10+OOFFSET] ;source2 offset
        mov     rdx,[r10+LOFFSET]
        shr     rdx,8            ;source2 length
        mov     rdi,[r8+AOFFSET] ;target address
        add     rdi,[r8+OOFFSET] ;target offset
        mov     rcx,[r8+LOFFSET]
        shr     rcx,8            ;target length

        mov     r10,8
SILXORFUNlp1:
        cmp     rbx,r10
        jl      SILXORFUNs      ;jump if short amount left
        cmp     rdx,r10
        jl      SILXORFUNs      ;jump if short amount left

        mov     rax,[rsi]
        XOR     rax,[r9]
        mov     [rdi],rax
        add     rsi,r10
        add     r9,r10
        add     rdi,r10
        sub     rbx,r10
        sub     rdx,r10
        sub     rcx,r10
        jmp     SILXORFUNlp1
SILXORFUNs:
        and     rbx,rbx
        jz      SILXORFUNfb     ;jump if one has run out
        and     rdx,rdx
        jz      SILXORFUNfd     ;jump if one has run out
        mov     al,[rsi]
        XOR     al,[r9]
        mov     [rdi],al
        inc     rsi
        inc     r9
        inc     rdi
        dec     rbx
        dec     rdx
        dec     rcx
        jmp     SILXORFUNs
SILXORFUNfb:
        jrcxz   SILXORFUNdone
        mov     rcx,rdx
        mov     rsi,r9
        rep movsb

SILXORFUNfd:
        jrcxz   SILXORFUNdone
        mov     rcx,rbx
        rep movsb

SILXORFUNdone:
        ret


; =============================================================================
; dumpdad: prints symbol version of address if possible
dumpdad PROC
        dbgsv
        cmp     rax,WorkSpaceAddr
        jb      dumpdadnwk
        cmp     rax,WorkSpaceLast
        jb      dumpdadwk               ;within work space
dumpdadnwk:
        lea     rsi,MapLbl0             ;get symbol table list
        mov     rcx,rsi                 ;save for end of list check
        cmp     rcx,0
        jz      dumpdaddone             ;no map, so cant do
        xor     rbx,rbx
        mov     r8,SIL100maxpos         ;distance to closest symbol at
        mov     r9,rsi
dumpdadlp:
        mov     rsi,[rsi]               ;go to next list entry
        cmp     rsi,rcx ;at end of list?
        je      dumpdadlpe              ;jump if at end of list
        cmp     rax,[rsi+8]             ;compare address
        je      dumpdadfoundx           ;jump if dount exact
        mov     rdx,rax
        sub     rdx,[rsi+8]             ;compute distance
        jb      dumpdadlp               ;don't use if symbol is higher than address
dumpdadnotn:
        cmp     rdx,r8                  ;compare to best prior
        jnl     dumpdadlp               ;not better
        mov     r8,rdx                  ;save distance
        mov     r9,rsi                  ;save entry location
        jmp     dumpdadlp

dumpdadfoundx:  xor r8,r8               ;exact find - distance is zero
        jmp     dumpdadfound
dumpdadlpe:                             ;at end of list
        mov     rsi,r9                  ;best entry found
dumpdadfound:
        add     rsi,17                  ;skip to symbol names, after first blank
        cmp     r8,4096
        ja      dumpdaddone             ;don't use symbol if distance >4096
dumpdadlp2:
        mov     al,[rsi]
        cmp     al,' '                  ;print up to blank
        je      dumpdaddonex
        call    dbgprtc
        inc     rsi
        jmp     dumpdadlp2
dumpdaddonex:
        cmp     r8,0
        je      dumpdaddone
        call    dbgs
        db      '+',0
        mov     rax,r8
        call    dbgprtx
        jmp     dumpdaddone

dumpdadwk:
        sub     rax,WorkSpaceAddr
        call    dbgs
        db      'wk+',0
        call    dbgprtx
dumpdaddone:
        dbgrs
        ret
dumpdad ENDP

dumparea proc
        dbgsv
dumparealp:
        mov     rax,rsi
        call    dbgprtx
        call    dbgs
        db ': ',0
        call    dbgckad
        cmp     rsi,rax
        jge     dumpareaskp
        mov     rax,[rsi]
        call    dbgprtx
dumpareaskp:
        call    dbgblnk
        call    dumpdad
        call    dbgcrlf
        add     rsi,8
        sub     rcx,8
        jnl     dumparealp
        dbgrs
        ret
dumparea endp

dumpdescr proc  ;dump descrptor pointed to by rsi
        dbgsv
        call dbgs
        db 'Dump D of descriptor=>',0
        mov rcx,DESCR
        call dbgdump
        call dbgs
        db 'A=>',0
        mov rax,[rsi]
        call dbgprtx
        call dbgblnk
        cmp MapLbl0,0
        je dumpdescrnd
        mov rax,[rsi]
        call dumpdad
        jmp dumpdescrok
dumpdescrnd:
        sub rax,WorkSpaceAddr
        call dbgs
        db 'A rel WS start=',0
        call dbgprtx
dumpdescrok:
        call dbgs
        db ' V&F=',0
        mov rax,[rsi+VOFFSET]
        call dbgprtx
        call dbgcrlf
        call dbgcrlf
        dbgrs
        ret
dumpdescr endp

dumpflt proc
        dbgsv
        fltsv

        fstp    dumpflt0
        fstp    dumpflt1
        fstp    dumpflt2
        fstp    dumpflt3
        fstp    dumpflt4
        fstp    dumpflt5
        fstp    dumpflt6
        fstp    dumpflt7

        lea     rsi,dumpfstring

        mov     rax,dumpflt0
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(0)'

        mov     rax,dumpflt1
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(1)'

        mov     rax,dumpflt2
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(2)'

        mov     rax,dumpflt3
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(3)'

        mov     rax,dumpflt4
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(4)'

        mov     rax,dumpflt5
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(5)'

        mov     rax,dumpflt6
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(6)'

        mov     rax,dumpflt7
        call    dbgprtx
        call    dbgblnk
        call    FLT2STR
        call    dbgstr
        dbgustr '=st(7)'

        call    dbgcrlf

        fld     dumpflt7
        fld     dumpflt6
        fld     dumpflt5
        fld     dumpflt4
        fld     dumpflt3
        fld     dumpflt2
        fld     dumpflt1
        fld     dumpflt0
        fltrs
        dbgrs
        ret
dumpflt endp
