; Macros to implement SNOBOL-like pattern matching in assembler
; Written by Viktors Berstis
;
;Pattern element rules:
; Entry into a pattern element is at the first code address
; If any state variable is needed, push the variables and use patstack
;    and use patstackx on failure out of the element so stack is restored
; On failure, each element should set cursoro back to value it saw on first entry
; priorprv should be set to branch point to go to from failures of succeeding element
; Note that to support ARBNO(), use of patstack is required in general.
; These work as if &ANCHOR = 1.  Use parb after pstart if you don't want that.

; See pattest.asm for example usage


;==============================================================================
; str is register or storage area or equ with address of subject string
; strl is register or storage area or equ with length of subject string or number
; stra is label for storage area containing the string
; istr is an immediate string e.g. 'hello'
; strmx is the max length string that can be stored at stra
; p is a sequence of balanced pattern macros
; fail is address of code to branch to if pattern fails to match

;Pattern elements:
; * = implemented
; *      PSTART   Start of pattern [str,strl,fail]
; *      PEND     End of pattern
; *      PSTR     Match given string [stra, strl]
; *      PSTRI    Match given string [istr]
; *      PLP      Left parenthesis  ( p
; *      PALT     Alternation       | p ...
; *      PRP      Right parenthesis )
; *      PASSNS   Start of assignment mark ( p
; *      PASSNE   End of assignment mark   ) [stra,strl,strmx]
; *      PLEN     String of length n [strl]
; *      PARB     Arbitrary length string try short to long strings
; *      PMAXARB  Arbitrary length string try long to short strings
; *      PBREAK   Match up to character in specified set [stra,strl]
; *      PBREAKI  Match up to character in specified set [istr]
; *      PSPAN    Match max length of characters in given set [stra,strl]
; *      PSPANI   Match max length of characters in given set [istr]
; *      PANY     Match one of any of the characters specified [stra,strl]
; *      PANYI    Match one of any of the characters specified [istr]
; *      PNOTANY  Match a character not in specified set [stra,strl]
; *      PNOTANYI Match a character not in specified set [istr]
; *      PBREAKX  Match up to successive break characters [stra,strl]
; *      PBREAKXI Match up to successive break characters [istr]
; *      PRPOS    Match position n from the right of the subject [strl]
; *      PPOS     Match position n from the left of the subject [strl]
; *      PREM     Match remainder of subject string
; *      PTAB     Match up to position n [strl]
; *      PRTAB    Match up to position n back from the end of the subject [strl]
; *      PARBNOS  Arbitrary number of instances of following pattern ARBNO(p
; *      PARBNOE  End of ARBNO                                              )
; *      PFAIL    Always fail
;        PBAL     Match parenthesis balanced string
;
;==============================================================================
;
;
;         Pattern element               Pattern element               Pattern element
;      +------------------+          +------------------+          +------------------+
;      |                  |          |                  |          |                  |
; ---->|nxt:            ok|---->---->|nxt:            ok|--[opt]-->|nxt:            ok|---->
;      |                  |          |                  |          |                  |
; <----|fail          prv:|<----<----|fail          prv:|<----<----|fail          prv:|<----
;      |                  |          |                  |          |                  |
;      +------------------+          +------------------+          +------------------+
;
;Forward flow of control is accomplished with consecutive instructions.
;[opt] is optional normal code.
;
;Globals:
;                cursoro         Offset pointer to current match position
;                subjecta        Address of subject string
;                subjectl        Length of subject string
;                stackpat        Place where pattern match saves initial rsp pointer
;
;Pattern element entry points:
;      nxt:      Try to match pattern element for the first time
;      priorprv: Try to match an alternative
;      others for more complicated pattern elements
;
;Pattern element exit points:
;      ok        Element matched, go to next element
;      fail      No more alternatives to match, go to prior element for an alternative
;
;Stack frame entries are really not a stack, but a thread through the pattern elements.
;
;==============================================================================


;==============================================================================
; PATVARS macro used to set up work data area for pattern matches
;==============================================================================
patvars macro
subjecta dq     ?       ;Subject string address
subjectl dq     ?       ;Subject string length
cursoro dq      ?       ;Cursor offset in subject string
cursormax dq    ?       ;Max cursor offset during match
stackpat dq     ?       ;Initial stack pointer
stacklvl dq     ?       ;Stack frame chain
priormark dq    ?       ;address in stack of prior pattern element start marker
marklevel dq    ?       ;mark level
plev =          0       ;Pattern nesting depth
patnum   =      0       ;Unique macro number generator
        endm

;==============================================================================
; Macros for managing pattern labels and nesting levels
;==============================================================================
patstack macro  a,b,c,d,e       ;;create new pattern frame on stack
        push    &a&&b&&c&&d&&e& ;;save prior frame
        push    stacklvl
        mov     stacklvl,rsp
        mov     &a&&b&&c&&d&&e&,rsp ;;set our frame
        endm
patstackx macro a,b,c,d,e       ;;exit pattern frame on stack to prior one
        mov     rsp,stacklvl    ;;maybe don't need to do this if rsp not moved ####
        mov     rax,[rsp+8]
        mov     &a&&b&&c&&d&&e&,rax ;;restore prior frame pointer
        pop     rsp
        mov     stacklvl,rsp
        endm

patinc  macro   p,q     ;;used by other macros to increment
ifndef  p&q&
p&q&    =       0
endif
p&q&    =       p&q& + 1
patcur  =       p&q&
        endm

patdec  macro   p,q     ;;used by other macros to decrement
ifndef  p&q&
p&q&    =       0
endif
p&q&   =        p&q& - 1
patcur  =       p&q&
        endm

plevinc macro                   ;;increment nesting level
ifndef  plev
plev    =       0
endif
plev    =       plev+1
        endm

plevdec macro                   ;;decrement nesting level
ifndef  plev
plev    =       0
endif
plev    =       plev-1          ;;current plev is in plev
        endm

patvar  macro   a,b,c,d,e       ;;create a temp variable for a pattern element
        .data
        align   8
&a&&b&&c&&d&&e& dq 0
        .code
        endm
                                ;;set temp variable from register r
patput  macro   r,a,b,c,d,e
        mov     &a&&b&&c&&d&&e&,&r&
        endm

patget  macro   r,a,b,c,d,e     ;;get temp variable value into register r
        mov     &r&,a&&b&&c&&d&&e&
        endm

patlea  macro   r,a,b,c,d,e     ;;get temp address into register r
        lea     &r&,a&&b&&c&&d&&e&
        endm

pnuminc macro                   ;;increment patern element number for plev
        patinc  pnum,%plev
pnumcur =       patcur
        endm

pnumdec macro                   ;;decrement patern element number for plev
        patdec  pnum,%plev      ;;pnum&plev = pnum&plev-1
pnumcur =       patcur
        endm

passinc macro                   ;;increment patern element number for passns
        patinc  pass,%plev
passcur =       patcur
        endm

passdec macro                   ;;decrement patern element number for passns
        patdec  pass,%plev
passcur =       patcur
        endm

pspinc  macro                   ;;increment patern element number for pspan
        patinc  psp,%plev
pspcur  =       patcur
        endm

plpinc  macro                   ;;increment patern element number for plp
        patinc  plp,%plev
plpcur  =       patcur
        endm

plpdec  macro                   ;;decrement patern element number for plp
        patdec  plp,%plev
plpcur  =       patcur
        endm

paninc  macro                   ;;increment patern element number for parbnos
        patinc  pan,%plev
pancur  =       patcur
        endm

pandec  macro                   ;;decrement patern element number for parbnos
        patdec  pan,%plev
pancur  =       patcur
        endm

paltinc macro                   ;;increment patern element number for palt
        patinc  palt,%plev
paltcur =       patcur
        endm

patlab  macro   p,q,r,s         ;;make label from variables
&p&q&r&s:
        endm

patjmp  macro   p,q,r,s         ;;jump to label
        jmp     &p&&q&&r&&s&
        endm

jmppriorprv macro
        jmp     priorprv
        endm

jmppatnxt macro PATNXT
        jmp     PATNXT
        endm

patcall macro   p,q,r,s
        call    &p&&q&&r&&s&
        endm

curmax  macro   regis
   ifb  <regis>
        push    rax
        mov     rax,cursoro
        cmp     rax,cursormax
        jl      @F
        mov     cursormax,rax
@@:
        pop     rax
   else
        cmp     regis,cursoro
        jna     @F
        mov     cursoro,regis
@@:
   endif
        endm

;==============================================================================
; PSTART        string,stringl,[failure]     ;Start of pattern
;               - string is the subject string to be matched
;               - stringl is the length of the subject string
;               - if failure branch address is not specified, we go to the end of the pattern
;==============================================================================

pstart  MACRO   STR,STRL,F
        LOCAL   FBAK,PATNXT
        mov     rsi,STR         ;load address of subject string
        mov     rcx,STRL        ;load length of subject string
        mov     cursoro,0       ;set current cursor offset to start of subject
        mov     cursormax,0     ;set max cursor offset to start of subject
        mov     subjecta,rsi    ;save address of subject string in pattern data area
        mov     subjectl,rcx    ;save length of subject string
        mov     stackpat,rsp    ;save stack pointer so can restore later
        mov     rax,rsp
        sub     rax,8
        push    rax
        mov     stacklvl,rax    ;top stack level pointer
        jmppatnxt PATNXT        ;go to next pattern element
FBAK:
        MOV     rsp,stackpat    ;restore stack pointer from start of match
        IFB     <F>
.ERR should not have null failure branch address on PSTART
;#### could branch to pattern end or something on failure
        ELSE
         JMP     F              ;jump to failure address
        ENDIF
PATNXT:
priorprv =    FBAK              ;set backtrack address for next pattern element
        ENDM

;==============================================================================
; PEND          [success]       ;End of pattern
;==============================================================================

pend    MACRO   success
        MOV     rsp,stackpat    ;restore stack pointer from start of match
        IFNB    <success>
         JMP    success         ;put in branch to success address if specified
        ENDIF
        ENDM

;==============================================================================
;       PARB    - match increasing string lengths
;==============================================================================

parb    MACRO
        LOCAL   FBAK,FBAKT,PATNXT,AFTC,RETRY
        plevinc
        pnuminc
        patvar  pat,%plev,frame,%pnumcur ;frame variable
        push    cursoro         ;arbcur +24
        xor     rax,rax
        push    rax             ;arblen +16
        patstack pat,%plev,frame,%pnumcur ;set frame
        xor     rcx,rcx
        patget  r11,pat,%plev,frame,%pnumcur
RETRY:  mov     [r11+16],rcx    ;save initial length to match
        mov     rax,cursoro     ;save starting cursor position on stack
        mov     [r11+24],rax
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,rax         ;calculate remaining unmatched length of subject
        sub     rbx,rcx         ;remaining length of subject if this string matches
        jb      FBAKT           ;fail if not enough characters in subject
        add     cursoro,rcx     ;new cursor offset
        curmax  rcx
AFTC:
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAKT:
        patstackx pat,%plev,frame,%pnumcur ;restore prior frame
        jmppriorprv             ;quit ARB, is too long now
FBAK:
        patget  r11,pat,%plev,frame,%pnumcur
        mov     rax,[r11+24]
        mov     cursoro,rax     ;on failure, restore cursor position and
        curmax  rax
        mov     rcx,[r11+16]
        inc     rcx
        jmp     RETRY           ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        plevdec
        ENDM

;==============================================================================
;       PMAXARB - match decreasing string lengths
;==============================================================================

pmaxarb MACRO
        LOCAL   FBAK,FBAKT,PATNXT,RETRY
        plevinc
        pnuminc

        patvar  pat,%plev,maframe,%pnumcur ;frame variable
        push    cursoro         ;arbcur +24
        mov     rcx,subjectl
        sub     rcx,cursoro     ;max length to try first
        push    rcx             ;arblen +16
        patstack pat,%plev,maframe,%pnumcur ;set frame
        patget  r11,pat,%plev,maframe,%pnumcur
RETRY:  cmp     rcx,0
        jl      FBAKT
        mov     [r11+16],rcx     ;save this length
        mov     rax,cursoro     ;save starting cursor position on stack
        mov     [r11+24],rax
        add     cursoro,rcx     ;new cursor offset
        curmax  rcx
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAKT:  patstackx pat,%plev,maframe,%pnumcur
        jmppriorprv             ;quit ARB, is too long now
FBAK:   patget  r11,pat,%plev,maframe,%pnumcur
        mov     rax,[r11+24]
        mov     cursoro,rax     ;on failure, restore cursor position and
        curmax  rax
        mov     rcx,[r11+16]
        dec     rcx
        jmp     RETRY           ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        plevdec
        ENDM

;==============================================================================
;       PREM    - match remainder of subject string
;==============================================================================

prem    MACRO
        LOCAL   FBAK,PATNXT
        plevinc
        pnuminc
        patvar  pat,%plev,frame,%pnumcur ;frame variable
        push    cursoro         ;arbcur +16
        patstack pat,%plev,frame,%pnumcur ;set frame
        mov     rbx,subjectl    ;get subject string length
        mov     cursoro,rbx     ;move cursor to end of subject
        curmax  rbx
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:   patget  r11,pat,%plev,frame,%pnumcur
        mov     rax,[r11+16]
        mov     cursoro,rax     ;on failure, restore cursor position and
        curmax  rax
        patstackx pat,%plev,frame,%pnumcur ;restore prior frame
        jmppriorprv             ;fail
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        plevdec
        ENDM

;==============================================================================
;       PTAB    - match up to a specified position in the subject string
;==============================================================================

ptab    MACRO   tabpos
        LOCAL   FBAK,PATNXT
        plevinc
        pnuminc
        patvar  pat,%plev,frame,%pnumcur ;frame variable
        push    cursoro         ;arbcur +16
        patstack pat,%plev,frame,%pnumcur ;set frame
        mov     rbx,tabpos      ;target position
        cmp     rbx,cursoro     ;make sure we are not past current position
        jl      FBAK
        mov     cursoro,rbx     ;move cursor to end of subject
        curmax  rbx
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:   patget  r11,pat,%plev,frame,%pnumcur
        mov     rax,[r11+16]
        mov     cursoro,rax     ;on failure, restore cursor position and
        curmax  rax
        patstackx pat,%plev,frame,%pnumcur ;restore prior frame
        jmppriorprv             ;fail
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        plevdec
        ENDM


;==============================================================================
;       PRTAB   - match up to a specified position relative to the end of the subject string
;==============================================================================

prtab   MACRO   tabpos
        LOCAL   FBAK,PATNXT
        plevinc
        pnuminc
        patvar  pat,%plev,frame,%pnumcur ;frame variable
        push    cursoro         ;arbcur +16
        patstack pat,%plev,frame,%pnumcur ;set frame
        mov     rbx,subjectl    ;get length of subject
        sub     rbx,tabpos      ;subtract from end of subject string
        cmp     rbx,cursoro     ;make sure we are not past current position
        jl      FBAK
        mov     cursoro,rbx     ;move cursor to end of subject
        curmax  rbx
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:   patget  r11,pat,%plev,frame,%pnumcur
        mov     rax,[r11+16]
        mov     cursoro,rax     ;on failure, restore cursor position and
        curmax  rax
        patstackx pat,%plev,frame,%pnumcur ;restore prior frame
        jmppriorprv             ;fail
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        plevdec
        ENDM


;==============================================================================
;       PLEN    stringl
;               - length of string to match against subject
;==============================================================================

plen    MACRO   STRL
        LOCAL   FBAK,PATNXT,FBAKL
        MOV     rcx,STRL        ;load length of string
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        sub     rbx,rcx         ;remaining length of subject if this string matches
        jb      FBAKL           ;fail if not enough characters in subject
        add     cursoro,rcx     ;save cursor
        curmax
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:
        mov     rax,cursoro     ;on failure, restore cursor position and
        sub     rax,STRL
        mov     cursoro,rax
        curmax  rax
FBAKL:
        jmppriorprv             ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

;==============================================================================
; PPOS Succeeds if offset is equal to argument
;==============================================================================
ppos    MACRO   POSIT
        LOCAL   FBAK,PATNXT,PATNXTX
        mov     rax,cursoro
        cmp     rax,POSIT
        je      PATNXTX
FBAK:
        jmppriorprv             ;go to prior pattern element for an alternative
PATNXTX: jmppatnxt PATNXT
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

;==============================================================================
; PRPOS Succeeds if offset is equal to argument
;==============================================================================
prpos   MACRO   POSIT
        LOCAL   FBAK,PATNXT,PATNXTX
        mov     rax,subjectl
        sub     rax,cursoro
        cmp     rax,POSIT
        je      PATNXT
FBAK:
        jmppriorprv             ;go to prior pattern element for an alternative
PATNXTX: jmppatnxt PATNXT
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

;==============================================================================
;       PSTR    string,stringl
;               - string is string to match against subject
;               - length of string to match against subject
;       PSTRI   'string'        ;immediate string form
;==============================================================================

pstr    MACRO   STR,STRL
        LOCAL   FBAK,PATNXT,AFTC,FBAKL
        LEA     rsi,STR         ;load address of string
        MOV     rcx,STRL        ;load length of string
        mov     rdi,subjecta    ;get subject string address
        add     rdi,cursoro     ;address in subject string at current cursor
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        sub     rbx,rcx         ;remaining length of subject if this string matches
        jb      FBAKL           ;fail if not enough characters in subject
        repz cmpsb              ;do the compare
        jnz     FBAKL           ;fail if does not compare
        sub     rdi,subjecta    ;calculate new cursor offset
        mov     cursoro,rdi     ;save cursor
        curmax  rdi
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:
        mov     rax,cursoro     ;on failure, restore cursor position and
        sub     rax,STRL
        mov     cursoro,rax
        curmax  rax
FBAKL:
        jmppriorprv             ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

pstri   MACRO   STRING
        LOCAL   AROUND,STR,STRL
        jmp     AROUND
STR     db      STRING
STRL    dq      $-STR
AROUND: pstr    str,strl
        ENDM

;==============================================================================
;       PARBNOS  pat  PARBNOE    x = ARBNO(pat)  is same as x = null | pat *x
;fail if last pat execution did not move cursor or if at end of subject.. ANCUR variable
;==============================================================================
;
;        PARBNOS                pat               PARBNOE
;        +-->-----1stTime-------------------------------+
;        |                                              |
;        |                      pat                     |ANEND
;      +-|-------+          +---------+         +-------v-+         +--
;      | |       |          |         |         |       | |         |
;PATNXT| |   var |    PATNXT|         |   PATNXT|       | |   PATNXT|
; ---->|-+     *------>---->|         |---->--->--------+------>--->|
;      |       | |          |         |         |         |         |
; <------*var--|-<-----<----|         |<-----------+    +--<--------|
;      | |     | |PATPRV    |         |PATPRV   |  |    | |PATPRV   |
;      +-|-----^-+          +---------+         +--^----|-+         +--
;        |     |ANMORE                         ANBK|    |
;        |     |                                   |    |
;        +->---|-----------------------------------+    |
;              +---<------InfiniteLoopTest?-------------+
;

;==============================================================================
;       PARBNOS  ;; not implemented
;==============================================================================

parbnos MACRO
        LOCAL   PATNXT,PATPRV,PATNXTX,PATQUIT
; ARBNO START =================================================================
        plevinc
        paninc
        patvar  pat,%plev,anframe,%pancur ;frame variable
        push    cursoro         ;ANCUR +24
        mov     rax,1
        push    rax             ;ANRT +16 depth count
PATQUIT:
        patstack pat,%plev,anframe,%pancur ;set frame     ;set frame
        patjmp  pat,%plev,anend,%pancur ;try pattern element after PARBNOE

PATPRV:
        patget  r11,pat,%plev,anframe,%pancur
        mov     rbx,[r11+16]    ;get depth count
        dec     rbx
        mov     [r11+16],rbx
        cmp     rbx,1
        ja      PATNXTX
        patstackx pat,%plev,anframe,%pancur ;pop stack
        jmppriorprv
PATNXTX: patjmp pat,%plev,anbk,%pancur

;anmore:
        patlab  pat,%plev,anmore,%pancur
        patget  r11,pat,%plev,anframe,%pancur
        mov     rbx,[r11+16]    ;get depth count
        inc     rbx
        mov     [r11+16],rbx
        cmp     rbx,1000000     ;check for "infinite" loop ####
        ja      PATQUIT
        jmppatnxt PATNXT        ;try another copy of pattern

priorprv =      PATPRV
PATNXT:
        ENDM

;==============================================================================
;       PARBNOE
;==============================================================================

parbnoe MACRO
        LOCAL   PATNXT,PATPRV
; ARBNO END ===================================================================
        paninc
        pandec                          ;update pancur
;anend:
        patlab  pat,%plev,anend,%pancur
        jmppatnxt PATNXT

PATPRV:
        patjmp  pat,%plev,anmore,%pancur ;try pattern element after PARBNOE

;anbk:
        patlab  pat,%plev,anbk,%pancur
        jmppriorprv

priorprv =      PATPRV
PATNXT:
        plevdec
        ENDM

;==============================================================================
;       PLP  pat  [PALT  pat]...  PRP     alternation
;==============================================================================
;         PLP                                      PALT                                    PALT                                     PRP
;                                                 +---------------------------------------*---------------------------------------*-----+
;                                                 |                                       |                                       |     |
;          (                    pat1              |  |                  pat2              |  |                  pat3              |  )  |ALTEND
;      +---------+          +---------+         +-|-------+         +---------+         +-|-------+         +---------+         +-|-----|-+         +--
;PATNXT|         |    PATNXT|         |   PATNXT| |       |   PATNXT|         |   PATNXT| |       |   PATNXT|         |   PATNXT| |     v |   PATNXT|
; ---->|-------------->---->|       ------->--->|-+   +-------->--->|         |---->--->|-+   +-------->--->|         |---->--->--+     +------>--->|
;      |         |          |         |         |     |   |         |         |         |     |   |         |         |         |         |         |
;      |         |          |         |         |    T|   |         |         |         |    T|   |         |         |         |         |         |
;      |         |          |         |         |    X|   |         |         |         |    X|   |         |         |         |         |         |
;      |         |          |         |         |    N|   |         |         |         |    N|   |         |         |         |         |         |
;      |         |          |         |         |    T|   |         |         |         |    T|   |         |         |         |         |         |
; <-------+    +--<----<----|         |<---<------+  L| +--<--------|         |<----------+  L| +--<--------|         |<----------------*--<--------|
;      |  |    | |PATPRV    |         |PATPRV   | |  A| | |PATPRV   |         |PATPRV   | |  A| | |PATPRV   |         |PATPRV   | +--+  | |PATPRV   |
;      +--^----|-+          +---------+         +-^---^-|-+         +---------+         +-^---^-|-+         +---------+         +-^--|--|-+         +--
;    LPAFT|    |                             ALTBK|   | |                            ALTBK|   | |                           ALTNXT|  |  |
;         |    +----------------------------------|---+ +---------------------------------|---+ +---------------------------------+  |  |
;         |                                       |                                       |                                          |  |
;         |                                       |                                       |                                          |  |
;         |                                       +---------------------------------------*---------------------------------------------+
;         |                                                                                                                          |
;         +--------------------------------------------------------------------------------------------------------------------------+
;
;
;==============================================================================
;============ ( left parenthesis
;==============================================================================
plp     MACRO
        LOCAL   PATNXT,PATPRV
; PLP PLP PLP =================================================================
        plevinc
        plpinc
        paltinc
        patvar  pat,%plev,lpframe,%plpcur ;frame variable
        push    rax             ;altvar +16 place keeper
        patstack pat,%plev,lpframe,%plpcur ;set frame     ;set frame
        jmppatnxt PATNXT        ;try pattern element after (

PATPRV:
        patinc  paltcur
        patjmp  pat,%plev,altnxt,%paltcur  ;jump to paltnxt
        patdec  paltcur

;lpaft:
        patlab  pat,%plev,lpaft,%plpcur ;all alternatives failed entry
        patstackx pat,%plev,lpframe,%plpcur
        jmppriorprv

priorprv =      PATPRV
PATNXT:
        ENDM
;==============================================================================
;============ | vertical bar
;==============================================================================
palt    MACRO
        LOCAL   PATNXT,PATPRV
; PALT PALT PALT ==============================================================
        paltinc
        plpinc
        plpdec
        patget  r11,pat,%plev,lpframe,%plpcur
        mov     rax,priorprv
        mov     [r11+16],rax     ;put into altvar
        patjmp  pat,%plev,altend,%plpcur

;altnxt:
        patlab  pat,%plev,altnxt,%paltcur  ;nxt case
        jmppatnxt PATNXT

PATPRV:
priorprv = PATPRV
        patlab  pat,%plev,altprv,%paltcur  ;prv case
        patinc  paltcur
        patjmp  pat,%plev,altnxt,%paltcur  ;jump to palt
        patdec  paltcur
PATNXT:
        ENDM
;==============================================================================
;============ ) right parenthesis
;==============================================================================
prp     MACRO   where
        LOCAL   PATNXT,PATPRV
; PRP PRP PRP =================================================================
        paltinc
        plpinc
        plpdec
        patget  r11,pat,%plev,lpframe,%plpcur
        mov     rax,priorprv
        mov     [r11+16],rax     ;put into altvar
        jmppatnxt PATNXT                ;this elements nxt entry

PATPRV:
priorprv = PATPRV
        patget  r11,pat,%plev,lpframe,%plpcur
        mov     rax,[r11+16]     ;get from altvar
        jmp     rax

;altnxt:
        patlab  pat,%plev,altnxt,%paltcur  ;nxt case
        patjmp  pat,%plev,lpaft,%plpcur    ;go to prior before left paren

;altend:
        patlab  pat,%plev,altend,%plpcur   ;just go to next pattern element
PATNXT:
        plevdec
        ENDM

;==============================================================================
; PFAIL Always fail
;==============================================================================
pfail   MACRO
        jmppriorprv
        ENDM

;==============================================================================
; PASSNS  pat  PASSNE var,varl,varmx    Assign matched string to string variable
;  Fails if string to be assigned is longer than varmx
;==============================================================================

passns  MACRO
        LOCAL   FBAK,PATNXT
        plevinc
        passinc
        push    cursoro ;+16    ;save starting cursor location
        patvar  pat,%plev,asframe,%passcur
        patstack pat,%plev,asframe,%passcur
        jmppatnxt PATNXT
FBAK:
        patstackx pat,%plev,asframe,%passcur
        jmppriorprv

priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

passne  MACRO   var,varl,varmx
        LOCAL   FBAK,PATNXT,PATNXTX
        passinc
        passdec                 ;make passcur correct
        patget  r11,pat,%plev,asframe,%passcur
        mov     rsi,[r11+16]    ;get cursor offset from passns
        mov     rcx,cursoro     ;cursor offset now
        sub     rcx,rsi         ;length of marked string
        add     rsi,subjecta
        cmp     rcx,varmx       ;compare to maximum allowed to be stored
        ja      FBAK            ;fail it too much to assign
        mov     varl,rcx
        lea     rdi,var
        jrcxz   PATNXTX
        rep movsb
PATNXTX: jmppatnxt PATNXT
FBAK:
        jmppriorprv
priorprv =      FBAK            ;failure entry from next pattern element
        plevdec                 ;Indicate one level shallower nesting number
PATNXT:
        ENDM

;==============================================================================
;       PANY    string,stringl
;               - string is characters to match against subject
;               - length of string
;       PANYI   'string'        ;immediate string form
;==============================================================================

pany    MACRO   STR,STRL
        LOCAL   FBAK,PATNXT,AFTC,PANYLP,NOTAN
        LEA     rsi,STR         ;load address of string
        MOV     rcx,STRL        ;load length of string
        jrcxz   NOTAN
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        cmp     rbx,1           ;remaining length of subject if this string matches
        jb      NOTAN           ;fail if not enough characters in subject
        mov     rdi,subjecta    ;get subject string address
        add     rdi,cursoro     ;address in subject string at current cursor
        mov     al,[rdi]        ;get character at cursor
PANYLP: cmp     al,[rsi]
        je      AFTC
        inc     rsi
        loop    PANYLP
        jmppriorprv             ;fail - none matched
AFTC:
        inc     cursoro         ;move cursor forward
        curmax
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:
        dec     cursoro         ;on failure, restore cursor position and
        curmax
NOTAN:  jmppriorprv             ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

panyi   MACRO   STRING
        LOCAL   AROUND,STR,STRL
        jmp     AROUND
STR     db      STRING
STRL    dq      $-STR
AROUND: pany    str,strl
        ENDM


;==============================================================================
;       PNOTANY string,stringl
;               - string is characters to not match against subject
;               - length of string
;       PNOTANYI 'string'       ;immediate string form
;==============================================================================

pnotany MACRO   STR,STRL
        LOCAL   FBAK,PATNXT,PANYLP,NOTAN
        LEA     rsi,STR         ;load address of string
        MOV     rcx,STRL        ;load length of string
        jrcxz   NOTAN
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        cmp     rbx,1           ;remaining length of subject if this string matches
        jb      NOTAN           ;fail if not enough characters in subject
        mov     rdi,subjecta    ;get subject string address
        add     rdi,cursoro     ;address in subject string at current cursor
        mov     al,[rdi]        ;get character at cursor
PANYLP: cmp     al,[rsi]
        je      NOTAN
        inc     rsi
        loop    PANYLP
        inc     cursoro         ;move cursor forward
        curmax
        jmppatnxt PATNXT        ;jump to next pattern element on success
FBAK:
        dec     cursoro         ;on failure, restore cursor position and
        curmax
NOTAN:  jmppriorprv             ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

pnotanyi MACRO   STRING
        LOCAL   AROUND,STR,STRL
        jmp     AROUND
STR     db      STRING
STRL    dq      $-STR
AROUND: pnotany str,strl
        ENDM

;==============================================================================
;       PSPAN   string,stringl
;               - string is characters to match against subject
;               - length of string
;       PSPANI  'string'        ;immediate string form
;==============================================================================

pspan   MACRO   STR,STRL
        LOCAL   FBAK,PATNXT,AFTC,PANYLP,PSPNLP,PSNF,PATNXTX
        pspinc
        patvar  pat,%plev,spframe,%pspcur
        push    cursoro         ;save cursor at start +16
        patstack pat,%plev,spframe,%pspcur
        LEA     rsi,STR         ;load address of string
        MOV     rcx,STRL        ;load length of string
        jrcxz   PSNF
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        jz      PSNF            ;fail if no characters in subject left to match
        mov     rdi,subjecta    ;get subject string address
        add     rdi,cursoro     ;address in subject string at current cursor
        mov     al,[rdi]        ;get character at cursor
PANYLP: cmp     al,[rsi]
        je      AFTC
        inc     rsi
        loop    PANYLP
PSNF:
        patstackx pat,%plev,spframe,%pspcur
        jmppriorprv             ;fail - none matched
AFTC:
        inc     cursoro         ;move cursor forward
        curmax
        inc     rdi
        dec     rbx             ;how much subject is left
        jz      PATNXTX
        lea     rsi,STR
        mov     rcx,STRL
        mov     al,[rdi]
PSPNLP: cmp     al,[rsi]
        je      AFTC
        inc     rsi
        loop    PSPNLP
PATNXTX: jmppatnxt PATNXT       ;jump to next pattern element on success
FBAK:
        patget  r11,pat,%plev,spframe,%pspcur
        mov     rax,[r11+16]    ;on failure, restore cursor position
        mov     cursoro,rax
        curmax  rax
        patstackx pat,%plev,spframe,%pspcur
        jmppriorprv             ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

pspani  MACRO   STRING
        LOCAL   AROUND,STR,STRL
        jmp     AROUND
STR     db      STRING
STRL    dq      $-STR
AROUND: pspan   str,strl
        ENDM

;==============================================================================
;       PBREAK  string,stringl
;               - string is characters break up to in subject
;               - length of string
;       PBREAKI 'string'        ;immediate string form
;==============================================================================

pbreak  MACRO   STR,STRL
        LOCAL   FBAK,PATNXT,AFTC,PANYLP,PSPNLP,PSNF,BRDONE
        pspinc
        patvar  pat,%plev,brframe,%pspcur
        push    cursoro         ;save starting cursor loc +16
        patstack pat,%plev,brframe,%pspcur
        LEA     rsi,STR         ;load address of string
        MOV     rcx,STRL        ;load length of string
        jrcxz   FBAK
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        jna     FBAK            ;fail if no characters in subject left to match
        mov     rdi,subjecta    ;get subject string address
        add     rdi,cursoro     ;address in subject string at current cursor
        mov     al,[rdi]        ;get character at cursor
PANYLP: cmp     al,[rsi]
        je      BRDONE
        inc     rsi
        loop    PANYLP
AFTC:
        inc     cursoro         ;move cursor forward
        curmax
        inc     rdi
        dec     rbx             ;how much subject is left
        jz      FBAK            ;didn't find break char in subject
        lea     rsi,STR
        mov     rcx,STRL
        mov     al,[rdi]
PSPNLP: cmp     al,[rsi]
        je      BRDONE
        inc     rsi
        loop    PSPNLP
        JMP     AFTC            ;look further, break char not seen yet
BRDONE:                         ;stop at break character
        jmppatnxt PATNXT        ;success
FBAK:
        patget  r11,pat,%plev,brframe,%pspcur  ;on failure, restore cursor position
        mov     rax,[r11+16]    ;get original cursor position
        mov     cursoro,rax
        curmax  rax
        patstackx pat,%plev,brframe,%pspcur
        jmppriorprv             ;go to prior pattern element for an alternative
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

pbreaki MACRO   STRING
        LOCAL   AROUND,STR,STRL
        jmp     AROUND
STR     db      STRING
STRL    dq      $-STR
AROUND: pbreak  str,strl
        ENDM


;==============================================================================
;       PBREAKX string,stringl
;               - string is characters break up to in subject and successive ones
;               - length of string
;       PBREAKXI 'string'        ;immediate string form
;==============================================================================

pbreakx MACRO   STR,STRL
        LOCAL   FBAK,FBAKE,PATMORE,PATGO,PATNXT,AFTC,PANYLP,PSPNLP,PSNF,BRDONE
        pspinc
        patvar  pat,%plev,brframe,%pspcur
        push    cursoro         ;save last break position loc +24
        push    cursoro         ;save starting cursor loc +16
        patstack pat,%plev,brframe,%pspcur
PATGO:
        LEA     rsi,STR         ;load address of string
        MOV     rcx,STRL        ;load length of string
        and     rcx,rcx
        jz      FBAKE
        mov     rbx,subjectl    ;get subject string length
        sub     rbx,cursoro     ;calculate remaining unmatched length of subject
        jna     FBAKE           ;fail if no characters in subject left to match
        mov     rdi,subjecta    ;get subject string address
        add     rdi,cursoro     ;address in subject string at current cursor
        mov     al,[rdi]        ;get character at cursor
PANYLP: cmp     al,[rsi]
        je      BRDONE
        inc     rsi
        loop    PANYLP
AFTC:
        inc     cursoro         ;move cursor forward
        curmax
        inc     rdi
        dec     rbx             ;how much subject is left
        jz      FBAKE           ;didn't find break char in subject
        lea     rsi,STR
        mov     rcx,STRL
        mov     al,[rdi]
PSPNLP: cmp     al,[rsi]
        je      BRDONE
        inc     rsi
        loop    PSPNLP
        JMP     AFTC            ;look further, break char not seen yet
FBAK:
        patget  r11,pat,%plev,brframe,%pspcur  ;on failure, restore cursor position
        mov     rbx,[r11+24]
        cmp     rbx,subjectl    ;have we run to the end of subject yet?
        jna     PATMORE
FBAKE:
        patget  r11,pat,%plev,brframe,%pspcur  ;on failure, restore cursor position
        mov     rax,[r11+16]    ;get original cursor position
        mov     cursoro,rax
        curmax  rax
        patstackx pat,%plev,brframe,%pspcur
        jmppriorprv             ;go to prior pattern element for an alternative
PATMORE: mov    cursoro,rbx
        curmax  rbx
        jmp     PATGO
BRDONE:                         ;stop at break character
        patget  r11,pat,%plev,brframe,%pspcur
        mov     rax,cursoro
        inc     rax
        mov     [r11+24],rax    ;save last found loc + 1
        jmppatnxt PATNXT        ;success
priorprv =      FBAK            ;failure entry from next pattern element
PATNXT:
        ENDM

pbreakxi MACRO  STRING
        LOCAL   AROUND,STR,STRL
        jmp     AROUND
STR     db      STRING
STRL    dq      $-STR
AROUND: pbreakx str,strl
        ENDM
