; Oregon SNOBOL5 source code
; Written by Viktors Berstis

; Single precision:
;    1 bit sign
;    8 bit biased exponent (this unsigned number - 127 = power of two)
;   23 bit significand (implied leading 1. bit if biased )

; Double precision:
;    1 bit sign
;   11 bit biased exponent (this unsigned number - 1023 = power of two)
;   52 bit significand (implied leading 1. bit)

; Extended precision:
;    1 bit sign
;   15 bit biased exponent (this unsigned number - 16383 = power of two)
;   64 bit significand (explicit leading 1. bit)

; Biased exponent of all zeros & significand not zero => denormalized
; Biased exponent of all ones => NAN or Infinity
; Infinity => zero significand
; NAN with high order significand bit on => Quiet NAN (quiet nan allowed to propagate)

; Floating point status register (from FSTSW...)
; msb 15     B  FPU busy
;     14     C3 condition
;     13-11  TOP of stack pointer
;     10     C2 condition
;      9     C1 condition
;      8     C0 condition
;      7     ES Error summary status
;      6     SF Stack fault
;      5     PE Precision error
;      4     UE Underflow error
;      3     OE Overflow error
;      2     ZE Zero divide error
;      1     DE Denormalized operand error
; lsb  0     IE Invalid operation error


; ============================================================================
;check if regis is a newly created NAN, insert stmt # if so
; regis must not be rax
nanck   macro   regis
        local   notnan,isinf,notlim
      ifdif   <regis>,<rax>
        push    rax
        mov     rax,regis
        cmp     rax,FMAX        ;is it almost infinity
        je      isinf
        cmp     rax,FMIN        ;is it almost -infinity
        je      isinf
        xor     rax,NaNoffexp
        and     rax,NaNoffexp
        jnz     notnan
        mov     rax,regis
        and     rax,NaNfrac
        jz      notnan
        mov     rax,regis
        and     rax,NaNindef
        jnz     notnan  ;already has a number
        mov     rax,STNOCL      ;get snobol stmt number
        cmp     rax,65536
        jb      notlim
        mov     rax,65535
notlim: shl     rax,1           ;don't use lowest bit
        and     rax,NaNindef    ;don't let it get too big
        or      regis,rax       ;insert statement number into NAN
        mov     rax,EXNOCL      ;get &STCOUNT
        shl     rax,17
        and     rax,NaNindef    ;don't let it get too big
        or      regis,rax       ;insert statement number into NAN
        jmp     notnan
isinf:  inc     rax             ;convert to infinity
        mov     regis,rax
notnan:
        pop     rax
      else
        error   mustnotuseraxinnanck
      endif
        endm

;NaNoffexp dq    7FF0000000000000h
;NaNfrac dq      000FFFFFFFFFFFFFh
;NaNindef dq     0007FFFFFFFFFFFFh


; =============================================================================
; Subroutine to convert real to string
; INPUT:
;    RSI    - Points to string buffer
;    RAX    - Real number to convert             (FLTW inside routine)
; OUTPUT:
;    RCX     - Length of string result
;
FLT2STR PROC
        fltsv                   ;save float state
        MOV     FLTW,RAX        ;Save input
        mov     fltbufst,rsi    ;Save buffer start location for length computation later
        OR      RAX,aoexexb     ;TURN on all but exponent bits
        NOT     RAX             ;CHECK FOR NAN/INF
        AND     RAX,RAX         ;Get condition code
        JNZ     FLT2NNI         ;JUMP IF NOT NAN/INF
; Number is either infinity or NAN
        MOV     RAX,FLTW        ;LOAD HIGH ORDER WORD
        AND     RAX,jsc         ;JUST SIGNIFICAND Bits remain
        JNZ     FLT2SNAN        ;JUMP IF NAN (significand not zero)
        mov     r8,FLTW
        TEST    r8,fltsgn       ;TEST FOR SIGN
        JZ      @F              ;JUMP IF POSITIVE
        MOV     BYTE PTR [RSI],'-' ;PREFIX WITH MINUS
        INC     RSI             ;Bump buffer pointer
@@:     MOV     WORD PTR [RSI+0],'NI'
        MOV     WORD PTR [RSI+2],'IF'
        MOV     WORD PTR [RSI+4],'IN'
        MOV     WORD PTR [RSI+6],'YT'
        ADD     RSI,8
        JMP     FLT2SALMOSTDONE
FLT2SNAN: MOV   WORD PTR [RSI+0],'AN'
        MOV     BYTE PTR [RSI+2],'N'
        ADD     RSI,3
;show statement number &STNO and statement count &STCONUT stored in NAN
        mov     rbx,FLTW        ;get the original NAN
        and     rbx,NaNindef
        shr     rbx,1
        JZ      FLT2SALMOSTDONE ;if stmt number is zero, then don't show
        and     rbx,0FFFFh      ;don't show stmt number greater than this
        mov     rdi,rsi
        call    INT2STR
        mov     byte ptr [rdi],':'
        inc     rdi
        mov     rbx,FLTW
        and     rbx,NaNindef
        shr     rbx,17          ;&STCOUNT is here
        jz      @F
        call    INT2STR
@@:     mov     rsi,rdi
        JMP     FLT2SALMOSTDONE

FLT2SISZR: ;ZERO IS THE NUMBER
        mov     r8,FLTW
        TEST    r8,fltsgn     ;TEST FOR SIGN
        JZ      @F              ;JUMP IF POSITIVE
        MOV     BYTE PTR [RSI],'-' ;PREFIX WITH MINUS
        INC     RSI
@@:     MOV     WORD PTR [RSI],'.0'
        ADD     RSI,2
        JMP     FLT2SALMOSTDONE

; Number is not NAN nor Infinity
FLT2NNI: mov    r8,FLTW
        TEST    r8,dnrlml ;TEST FOR DENORMAL (exponent zero, significand not zero)
        JNZ     FLT2SNDEN
        TEST    r8,jsc          ;test for zero significand
        JNZ     FLT2SDEN        ;jump if is denormal
        JMP     FLT2SISZR       ;jump if is zero
; Handle denormalized number here
FLT2SDEN:       ;treat denormal same as regular

FLT2SNDEN:      ;NOT DENORMALIZED NUMBER
        FLD     QWORD PTR FLTW  ;Number to convert
        WAIT
;dbgustr 'point 01 ####'
;call dbgfregs
        FMUL    FLTADJUST       ;get rid of .99999... #### is there a better way?
        WAIT
;dbgustr 'point 02 ####'
;call dbgfregs
        FLD     L2TENI          ;1.0/LOG2(10)
        WAIT
;dbgustr 'point 03 ####'
;call dbgfregs
        FLD     QWORD PTR FLTW  ;Number to convert
        WAIT
;dbgustr 'point 04 ####'
;call dbgfregs
        FABS                    ;needed for FYL2X operation below
        WAIT
;dbgustr 'point 03b####'
;call dbgfregs
        FMUL    FLTADJUST       ;get rid of .99999... ####
        WAIT
;dbgustr 'point 05 ####'
;call dbgfregs
 ;##### orig value cannot be negative or zero
        FYL2X                   ;get log base 10 of original number
        WAIT
;dbgustr 'point 06 ####'
;call dbgfregs
        fst     logFLTW
        wait
        fstcw   word ptr sil127fcw ;get float control word
        mov     ax,word ptr sil127fcw
        or      ax,0C00h        ;set rounding towards zero (truncate)
        mov     word ptr sil127fcw+2,ax
        fldcw   word ptr sil127fcw+2 ;load float control word
        FISTP   DWD4            ;Store power of ten, must truncate fraction
;showregi DWD4
;wait
;dbgustr 'point 10 ####'
;call dbgfregs
        fldcw   word ptr sil127fcw ;Restore rounding mode
        WAIT
        test    byte ptr logFLTW+7,80h  ;check sign of log of FLTW
        jz      @F
        dec     DWD4            ;decrement integer portion if log is negative
@@:     mov     FLTESW,0        ;Say not using E notation yet
        mov     rax,DWD4
        mov     FLTEEXP,rax     ;Save exponent in case we need to use e notation
;showregi FLTEEXP
        cmp     rax,0           ;Compute absolute value of exponent
        jge     @F
        neg     rax
@@:     cmp     rax,FLTDEC      ;have we exceeded &FLTDEC
        jle     @F
        mov     FLTESW,1        ;Say using E notation
@@:
        MOV     rbx,FLTSIG      ;Keep this many significant digits
        sub     rbx,1           ;Internally 0...n instead of 1...n
        jge     @F              ;don't let it go negative
        xor     rbx,rbx
@@:     cmp     rbx,14          ;don't let it go higher than 14 here (&FLTSTR = 15)
        jle     @F
        mov     rbx,14
@@:
        SUB     rbx,DWD4        ;Shift this power of ten to make sig digits an integer
        MOV     DWD4,rbx        ;Save difference
;showregi dwd4
        ; faster multiply by 10:
        ADD     rbx,rbx         ;*2
        MOV     rax,rbx         ;Save *2
        ADD     rbx,rbx         ;*4
        ADD     rbx,rbx         ;*8
        ADD     rbx,rax         ;*10    *length of TENSHFT elements
        WAIT
        lea     r10,TENSHFT
        FLD     TBYTE PTR [r10+rbx] ;Multiply by correct power of ten
        WAIT
        FMULP   ST(1),ST        ;shift significant digits as integer
        WAIT
        fldcw   word ptr sil127fcw+2 ;load float control word to truncate
        FBSTP   PKDVAL          ;Store packed decimal integer result
        fldcw   word ptr sil127fcw ;Restore rounding mode
        WAIT
; SET SIGN
        lea     r9,PKDVAL
        TEST    BYTE PTR [r9+9],80H    ;Check sign
        JZ      @F              ;Jump if positive
        MOV     BYTE PTR [RSI],'-' ;Prefix result with minus sign
        INC     RSI
@@:

; Count how many significant digits there are
        MOV     RAX,18          ;At most 18 digits
        MOV     rbx,8           ;offset to 9th byte in packed integer
FLT2SSC: lea    r9,PKDVAL
        CMP     BYTE PTR [r9+rbx],0 ;are both digits there zero?
        JNE     FLT2SNZ         ;jump if at least one not zero
        SUB     RAX,2           ;Two more zeros
        SUB     rbx,1           ;look at prior byte next
        JNL     FLT2SSC         ;Loop til done
        MOV     RAX,1           ;Just say there is one digit (when result is zero)
        JMP     FLT2SNZ1
FLT2SNZ: lea    r9,PKDVAL
        CMP     BYTE PTR [r9+rbx],11 ;is there a  zero there?
        JNC     FLT2SNZ1        ;jump if both digits in byte are not zero
        DEC     RAX             ;One fewer digits
FLT2SNZ1:
;       rax now contains number of significant digits in PKDVAL
        MOV     rbx,rax         ;Save number of significant digits in rbx
        SUB     RAX,DWD4        ;Compute number of digits left of decimal point
        cmp     fltesw,1        ;if need e format
        jne     @F              ;jump if not e format
        mov     rcx,1           ;force one leading digit in e format
        jmp     FLT2WSWDL       ;go handle e format version
@@:
        cmp     rax,0           ;check number of digits left of decimal point
        JE      FLT2SPTN        ;Starts with 0.nnnnn
        JG      FLT2SWSD        ;Starts with significant digits left of decimal

; Starts with leading 0.00...nnn ============================================
        NEG     RAX             ;make positive
        MOV     RCX,RAX         ;This many zeros after decimal point
        MOV     BYTE PTR [RSI],'0' ;STARTS WITH 0.
        MOV     BYTE PTR [RSI+1],'.'
        ADD     RSI,2           ;Account for "0." in result
FLT2SLZI: MOV   BYTE PTR [RSI],'0' ;Add a zero
        INC     RSI             ;Next result address
        LOOP    FLT2SLZI        ;Loop until inserted all leading zeros
        AND     rbx,rbx         ;See if any significant digits there
        JZ      FLT2SDN         ;Jump if number was all zeros
        MOV     RCX,rbx         ;Emit this many remaining significant digits
        JMP     FLT2WSWDK       ;Jump to do that

FLT2SPTN:  ;STARS with 0.nnnnn  =============================================
        MOV     BYTE PTR [RSI],'0' ;Starts with "0."
        MOV     BYTE PTR [RSI+1],'.'
        ADD     RSI,2           ;Account for the above
        AND     rbx,rbx         ;See if any significant digits there to add
        JZ      FLT2SDN         ;Jump if not
        MOV     RCX,rbx         ;Emit this many remaining significant digits
        JMP     FLT2WSWDK

FLT2SWSD:  ;STARTS with nnnn.nnn ============================================
        MOV     RCX,RAX         ;This many digits before decimal
FLT2WSWDL:                      ;Come here to handle e format case rcx=1
; Assumes rbx has count of significant digits, FLT2SEMIT decrements this
        CALL    FLT2SEMIT       ;Emit a digit
        LOOP    FLT2WSWDL       ;Do enough up to decimal
        MOV     BYTE PTR [RSI],'.'
        INC     RSI
        AND     rbx,rbx         ;See if any significant digits left to emit
        JZ      FLT2SDN         ;Jump if not
        MOV     RCX,rbx         ;This many left to do
FLT2WSWDK:
        CALL    FLT2SEMIT       ;Emit a digit
        LOOP    FLT2WSWDK       ;Do remaining significant digits
FLT2SDN:
        mov     rax,fltbufst    ;Get original RSI address (start of output area)
        MOV     rbx,RSI         ;Trim trailing zeros
FLT2SDNTL:
        DEC     rbx             ;look at output from right to left
        CMP     rax,rbx         ;Did we run all the way to the start?
        JNC     FLT2SDNT        ;jump if yes
        CMP     BYTE PTR 0[rbx],'0' ;Is this another trailing zero?
        JNE     FLT2SDNT        ;jump to look at next one to the left
        MOV     RSI,rbx         ;update end of output area pointer
        JMP     FLT2SDNTL       ;loop until no more trailing zeros
FLT2SDNT:
        cmp     fltesw,1        ;are we doing e format?
        jne     FLT2SALMOSTDONE
        mov     byte ptr [rsi],'e' ;Put out "e" for e format
        inc     rsi
; now output the exponent value in rax
        mov     rbx,FLTEEXP     ;get original exponent
        mov     rdi,rsi
        call    INT2STR
        mov     rsi,rdi

FLT2SALMOSTDONE:
        MOV     RCX,RSI         ;address after last+1 character
        SUB     RCX,fltbufst    ;Length of result string
        MOV     RSI,fltbufst    ;restore start rsi address
        fltrs                   ;restore float status
        RET
FLT2STR ENDP

; =============================================================================
; Emit bx'th digit to [RSI] and increment RSI and possibly decrement rbx
FLT2SEMIT PROC
        AND     rbx,rbx ;Are there more siginificant digits available?
        JZ      FLT2SEZ ;Emit zero instead because no more digits
        PUSH    RAX     ;Save rax to restore at end
        DEC     rbx     ;Decrement significant digit count
        PUSH    rbx
        SHR     rbx,1   ;Divide by 2
        lea     r9,PKDVAL
        MOV     AL,BYTE PTR [r9+rbx] ;Byte containing digit in packed data
        JNC     FLT2SENS ;jump if it was not the high order digit
        SHR     AL,4    ;move high order digit into low part of register
FLT2SENS:
        AND     AL,0FH  ;clear out excess bits
        OR      AL,30H  ;Turn into digit character
        MOV     [RSI],AL ;Put it in output area
        INC     RSI     ;Update output area address
        POP     rbx     ;restore remaining significant digit count
        POP     RAX     ;restore rax
        RET
FLT2SEZ:
        MOV     BYTE PTR [RSI],'0'      ;emit a zero digit (no significant ones remain)
        INC     RSI
        RET
FLT2SEMIT ENDP

; =============================================================================
; INITIALIZE POWERS OF TEN TABLE
FLTABLE PROC
; Initialize powers of ten, positives
        MOV     RAX,10
        MOV     DWD6,RAX        ;TEN
        WAIT
        FILD    DWD6            ;Load 10
        WAIT
        FLD1                    ;Load 1
        XOR     RBX,RBX
FLTABLP: WAIT
        lea     r10,TENSHFT
        FSTP    TBYTE PTR [r10+RBX] ;Store value into table
        WAIT
        FLD     TBYTE PTR [r10+RBX] ;Get value back
        ADD     RBX,10          ;Size of float long
        WAIT
        FMUL    ST,ST(1)        ;Multiply by 10
        CMP     RBX,340*10      ;Are we done?
        JL      FLTABLP
        WAIT
        FFREE   ST(0)           ;Throw away power of ten
        WAIT
        FINCSTP                 ;Pop it
        WAIT

; Now do negative powers of ten
        FLD1                    ;Load 1
        XOR     RBX,RBX
FLTABLN: WAIT
        FDIV    ST,ST(1)        ;Divide by 10
        SUB     RBX,10          ;Size of float long
        WAIT
        lea     R10,TENSHFT
        FSTP    TBYTE PTR [r10+RBX] ;Store value into table
        WAIT
        FLD     TBYTE PTR [r10+RBX] ;Get value back
        CMP     RBX,-(338*10)   ;Are we done?
        JNC     FLTABLN
        WAIT
        FFREE   ST(0)           ;Throw away power of ten
        WAIT
        FINCSTP                 ;Pop
        WAIT
        FFREE   ST(0)           ;Throw away ten
        WAIT
        FINCSTP                 ;Pop
        RET
FLTABLE ENDP

; Used in parsing a float number in a state machine
; Macro for parsing a character and then going to the next state in the state machine
PARSEANDGO MACRO TABLE
        CALL    S2FPRS          ;Parse a character and set transition offset
        lea     r11,TABLE       ;table of transitions for state TABLE
        jmp     qword ptr [r11+rbx] ;jump to next state
        ENDM

; =============================================================================
; Subroutine to convert string to real number
; INPUT:
;    RSI    - Points to string buffer
;    RCX    - Length of string result
; OUTPUT:
;    RAX    - Zero if ok
;    RDX    - Real number result            (DWD0 inside routine)
;
STR2FLT PROC
        AND     ECX,ECX         ;Check for null string
        JZ      S2FERROR        ;RETURN error if null #### or S2FZ ????
; Check for NAN
        CMP     RCX,3
        JNE     S2FNN1          ;NOT NAN
        CMP     WORD PTR [rsi],'aN'
        JE      @F
        CMP     WORD PTR [rsi],'AN'
        JNE     S2FNN1
@@:     CMP     BYTE PTR [rsi+2],'N'
        JNE     S2FNN1
        mov     r8,FNAN
        MOV     DWD0,r8         ;RETURN A QUIET NEGATIVE NAN
        SUB     RAX,RAX
S2FR:   MOV     RDX,DWD0
        RET

S2FZ:   MOV     RDX,FZERO      ;RETURN A ZERO
        XOR     RAX,RAX
        RET

S2FNN1: MOV     NEGSW,0         ;INDICATE POSITIVE
        CMP     BYTE PTR [RSI],'-' ;CHECK FOR MINUS
        JE      S2FN
        CMP     BYTE PTR [RSI],'+' ;CHECK FOR PLUS
        JNE     S2FNS
        JMP     S2FP

S2FERROR: MOV   RAX,1           ;NO DIGITS FOUND OR OTHER ERROR
        AND     RAX,RAX
        RET

S2FN:   MOV     NEGSW,80H
S2FP:   INC     rsi
        DEC     rcx
S2FNS:  JRCXZ   S2FERROR        ;JUMP IF NO DIGITS
; CHECK FOR INFINITY
        CMP     rcx,8
        JNE     S2FNI
        CMP     WORD PTR [rsi],'NI'
        JNE     S2FNI
        CMP     WORD PTR [rsi+2],'IF'
        JNE     S2FERROR
        CMP     WORD PTR [rsi+4],'IN'
        JNE     S2FERROR
        CMP     WORD PTR [rsi+6],'YT'
        JNE     S2FERROR
S2FINF: XOR     rax,rax
        mov     r9,FINF
        MOV     DWD0,r9
        mov     bl,NEGSW
        OR      BYTE PTR DWD0+7,bl
        JMP     S2FR

S2FNI:                          ;not infinity or nan
; Start parse of the float number here:
        SUB     rdx,rdx         ;Number of significant digits to left of decimal
        xor     r8,r8           ;init how many digits inserted in pkdval
        mov     r9,9            ;init destination offset within pkdval
        mov     s2fexpsign,'+'  ;assume positive exponent (after e) for now
        xor     r12,r12         ;exponent value after e
        MOV     AL,NEGSW
        MOV     BYTE PTR [PKDVAL+9],AL  ;Set decimal sign
        SUB     rax,rax
        MOV     QWORD PTR [PKDVAL],RAX  ;zero out first 8 bytes
        MOV     WORD PTR [PKDVAL+8],AX  ;zero out two more
        PARSEANDGO S2FS1B       ;Get character for state 1 of state machine

; Float number parsing state machine: (initial sign character already done)
; Addresses in each state mean what to do when:
;       0       zero character encountered
;       8       1-9 digit encountered
;      16       decimal point encountered
;      24       end of input string encountered
;      32       other kind of character encountered
;      40       "e" or "E" character encountered
;      48       "+" or "-" character encountered
        ALIGN   8
; State 1: looking for one leading digit
S2FS1B  DQ      S2FS2    ;Go to state 2 on zero
        DQ      S2FS3    ;Go to state 3 on 1-9
        DQ      S2FERROR ;Error for others
        DQ      S2FERROR
        DQ      S2FERROR
        DQ      S2FERROR
        DQ      S2FERROR

; State 2: consume leading zeros
S2FS2:
        PARSEANDGO S2FS2B
S2FS2B  DQ      S2FS2    ;Go to state 2 on zero
        DQ      S2FS3    ;Go to state 3 on 1-9
        DQ      S2FS4A   ;Go to state 4 on decimal
        DQ      S2FERROR ;Error for end of string
        DQ      S2FERROR ;Error for bad character
        DQ      S2FS5    ;Go to state 5 on e or E
        DQ      S2FERROR ;Error if sign character

; State 3: consume significant digits to left of decimal
S2FS3:
        CALL    S2FPUT          ;PUT DIGIT
        INC     rdx             ;ONE MORE DIGIT TO LEFT OF DECIMAL
        PARSEANDGO S2FS3B
S2FS3B  DQ      S2FS3    ;Go to state 3 on zero
        DQ      S2FS3    ;Go to state 3 on 1-9
        DQ      S2FS4A   ;Go to state 4 on decimal (don't store as digit)
        DQ      S2FS8    ;End of string, convert integer to real
        DQ      S2FERROR ;Error for bad character
        DQ      S2FS5    ;Go to state 5 if e or E
        DQ      S2FERROR ;Error if sign character

; State 4: handle characters after decimal
S2FS4:
        CALL    S2FPUT
S2FS4A: PARSEANDGO S2FS4B
S2FS4B  DQ      S2FS4    ;Go to state 4 on zero
        DQ      S2FS4    ;Go to state 4 ON 1-9
        DQ      S2FERROR ;Error if decimal
        DQ      S2FS8    ;Go to state 8 when done
        DQ      S2FERROR ;Error for bad character
        DQ      S2FS5    ;Go to state 5 on e or E
        DQ      S2FERROR ;Error if sign character

; State 5: handle characters after e or E
S2FS5:
        PARSEANDGO S2FS5B
S2FS5B  DQ      S2FS7    ;Go to state 7 on zero
        DQ      S2FS7    ;Go to state 7 ON 1-9
        DQ      S2FERROR ;Error if decimal
        DQ      S2FERROR ;Error if end of string
        DQ      S2FERROR ;Error for bad character
        DQ      S2FERROR ;Error if another e or E
        DQ      S2FS6    ;Go to state 6 on sign

; State 6: handle characters after sign after e or E
S2FS6:
        CMP     al,'-'   ;check for negative exponent sign
        jne     @F
        mov     s2fexpsign,'-'
@@:     PARSEANDGO S2FS6B
S2FS6B  DQ      S2FS7    ;Go to state 7 on zero
        DQ      S2FS7    ;Go to state 7 ON 1-9
        DQ      S2FERROR ;Error if decimal
        DQ      S2FS8    ;Go to state 8 when done
        DQ      S2FERROR ;Error for bad character
        DQ      S2FERROR ;Error for another e or E
        DQ      S2FERROR ;Error if another sign character

; State 7: handle exponent digits
S2FS7:  cmp     r12,500 ;if exponent is already too large
        jg      @F      ;skip this and don't make it bigger
        push    rax
        push    rbx
        push    rdx
        mov     ah,0
        mov     bx,ax
        mov     rax,r12
        mul     word10  ;multiply prior value by 10
        add     ax,bx   ;add in the new digit
        mov     r12,rax
        pop     rdx
        pop     rbx
        pop     rax
@@:
        PARSEANDGO S2FS7B
S2FS7B  DQ      S2FS7    ;Go to state 7 on zero
        DQ      S2FS7    ;Go to state 7 ON 1-9
        DQ      S2FERROR ;Error if decimal
        DQ      S2FS8    ;Go to state 8 when done
        DQ      S2FERROR ;Error for bad character
        DQ      S2FERROR ;Error for another e or E
        DQ      S2FERROR ;Error for sign character

; State 8: finish processing float value
S2FS8INF: JMP   S2FINF

S2FS8ZER: mov   rax,FZERO       ;RETURN ZERO
        OR      rax,QWORD PTR NEGSWD    ;GIVE PROPER SIGN
        MOV     DWD0,rax
        xor     rax,rax
        JMP     S2FR

S2FS8:
        SUB     rdx,18  ;SUBTRACT FOR 18 DIGITS
        CMP     s2fexpsign,'-'
        jne     @F
        neg     r12      ;negative if exponent had a minus sign
@@:     add     rdx,r12  ;add exponent after the e
        CMP     rdx,308  ;SEE IF SHOULD BE INFINITY      ####
        JG      S2FS8INF
        CMP     rdx,-324 ;SEE IF SHOULD BE ZERO          ####
        JL      S2FS8ZER
        WAIT
        FBLD    PKDVAL ;FBLD - LOAD DECIMAL VALUE
        MOV     rbx,rdx
        ADD     rbx,rbx ;*2
        MOV     rax,rbx ;save *2
        ADD     rbx,rbx ;*4
        ADD     rbx,rbx ;*8
        ADD     rbx,rax ;*10
                        ;rbx is now proper index into power table
        WAIT
        lea     r11,TENSHFT
        FLD     TBYTE PTR [r11+RBX] ;FMUL - ADJUST BY PROPER POWER OF TEN
        WAIT
        FMULP   ST(1),ST
        WAIT
        FSTP    QWORD PTR DWD0    ;FSTP - STORE RESULT
        WAIT
        mov     bl,NEGSW
        OR      BYTE PTR DWD0+7,bl      ;insert sign
        SUB     rax,rax
        JMP     S2FR

STR2FLT ENDP


; =============================================================================
; S2FPUT - PUT A CHARACTER IN PACKED DECIMAL FIELD
; INPUT:
;       AL - DECIMAL DIGIT TO INSERT
;       R8 - number of significant digits put into packed decimal field
;       R9 - byte offset into pkdval, initially 9
; OUTPUT:
;       PKDVAL - digit inserted in proper position
;       R8 - incremented
;       R9 - decremented every other time
S2FPUT  PROC
        cmp     r8,18   ;See if 18 digits already consumed
        jnc     S2FPUTD
        inc     r8      ;Say one more processed
        test    r8,1    ;See if odd
        jz      S2FPUTE ;Jump if even
        dec     r9      ;Decrement byte offset
        shl     al,4    ;Shift digit into high order position
        lea     r11,PKDVAL
        mov     byte ptr [r11+r9],al
        ret
S2FPUTE:
        lea     r11,PKDVAL
        or      byte ptr [r11+r9],al ;Put in lower order digit
S2FPUTD: ret
S2FPUT  ENDP


; =============================================================================
;Parse one character
; INPUT:
;       rcx       - Remaining length of input string
;       [rsi]     - Character
; OUTPUT:
;       rbx    - Result code
;               0 - Zero Digit, AL=0
;               8 - 1-9 Digit, AL=N
;               16- Decimal point
;               24- End of string
;               32- Bad character
;               40- e or E character
;               48- + or - character
;       rsi      - Incremented
;       rcx      - Decremented IF rbx!=24
;       al      - 0-9 IF rbx=0 or 8
S2FPRS  PROC
; dbgustr 'entering s2fprs ####'
        jrcxz   S2FPRSX ;jump if no more characters
        mov     al,[rsi] ;get the next character
; call dbgprtc
; dbgustr '=character ####'
        inc     rsi     ;increment address for next time
        dec     rcx     ;decrement remaining character count
        cmp     al,'.'  ;is it a decimal point
        je      S2FPRSD ;jump if so
        cmp     al,'e'  ;is it the "e" character
        je      S2FPRSE ;jump if so
        cmp     al,'E'  ;is it the "E" character
        je      S2FPRSE ;jump if so
        cmp     al,'+'  ;is it the "+" character
        je      S2FPRSS ;jump if sign
        cmp     al,'-'  ;is it the "-" character
        je      S2FPRSS ;jump if sign
        sub     AL,'0'  ;compare to zero character
        je      S2FPRSZ ;jump if zero
        jc      S2FPRSR ;jump if bad character
        cmp     al,10   ;is it not a digit
        jnc     S2FPRSR ;jump if bad character
        mov     rbx,8   ;1-9 DIGIT CODE
        ret
S2FPRSZ: mov    rbx,0   ;Zero character
        ret
S2FPRSD: mov    rbx,16  ;Decimal point code
        ret
S2FPRSX: mov    rbx,24  ;End of string code
        ret
S2FPRSR: mov    rbx,32  ;ERROR - Bad character
        ret
S2FPRSE: mov    rbx,40  ;"E" or "e" Character
        ret
S2FPRSS: mov    rbx,48  ;"+" or "-" sign character
        ret
S2FPRS  ENDP
