;Debug macros and subroutines for flat 64 bit addressing (linux	and windows)
;
;+-- x=MACRO rather than CALL, *=not implemented yet, -=partially implemented
;|
;| NAME	    DESCRIPTION
;| ----	    -----------
;x dbgsv    saves all registers	and flags on stack
;x dbgrs    restores all registers and flags on	stack
;x fltsv    saves all float registers etc
;x fltrs    restores all float registers etc
;  dbgprtc  print character in AL to standard output
;  dbgcrlf  print carriage return and line feed
;  dbgblnk  print blank
;  dbgb	    print blank
;  dbgs	    print characters after call	(up to zero byte)
;  dbgs16   print characters after call	(up to zero byte) skips	16 more	bytes
;  dbgprtd  print value	in AL in 3 digit unsigned decimal
;  dbgprtx1 print value	in AL in 2 digit hexadecimal
;  dbgprtx2 print value	in AX in 4 digit hexadecimal
;  dbgprtx4 print value	in EAX in 8 digit hexadecimal
;  dbgprtx8 print value	in RAX in 16 digit hexadecimal
;  dbgprtx10 print tenbyte at rsi in hex, right	to left	(useful	for dbl-ext-float)
;  dbgprtx  print value	in RAX in 16 digit hexadecimal
;  dbgregs  print the usable registers in hex
;  dbgstr   print string at RSI	for RCX	bytes
;  dbgstrz  print string at RSI	up to zero byte
;  dbgfrom  print 64bit	return address of who called the routine at entry
;  dbgdump  print storage at RSI for RCX bytes
;  dbgstack print storage at RSP for RCX bytes
;  dbgexit  exit from program
;  dbgitos  convert rax	two's complement to signed decimal string in dbgsnum
;  dbgprtfi print rax as formatted variable length decimal (using dbgitos)
;  dbgckad  check if address RSI is valid, returns max+1 address of region in RAX
;  dbggle   Windows GetLastError  - only for Windows
;x dbgistr  print immediate string as 5	parameters
;x dbgustr  print immediate string with	crlf
;  dbgqword print variable qword value
;
;* dbgfregs print the floating point registers in hex (see dumpflt in silcode.inc)
;* dbgdmpunit dump unit	area at	EDI formatted
;* dbgi	    debug interactive mode commands:
;		D [hexaddr [len	in hex]]	Display
;		Q				Quit debug call	(return	from rbgi)
;*		2				Toggle to dual output std+err
;		R				Show all registers
;*		S				Search memory for a string
;		X				Exit the whole program
;		?				Show command list

	dbgsv	MACRO
	PUSHFQ
	PUSH	RAX
	PUSH	RBX
	PUSH	RCX
	PUSH	RDX
	PUSH	RDI
	PUSH	RSI
	PUSH	R8
	PUSH	R9
	PUSH	R10
	PUSH	R11
	PUSH	R12
	PUSH	R13
	PUSH	R14
	PUSH	R15
	PUSH	RBP
	MOV	QWORD PTR [dbgRSPWK],RSP ;RSP to be restored on	DBGRS
	PUSHFQ
	PUSHFQ			    ;add 16 to RSP without modifying flags
	PUSHFQ
	MOV	QWORD PTR [dbgRSPWK2],RSP
	AND	BYTE PTR [dbgRSPWK2],0F0H ;make	it 16 byte aligned
	POPFQ			    ;retore flags after	AND instruction
	MOV	RSP,QWORD PTR [dbgRSPWK2]
	LEA	RSP,[RSP+8]	    ; make 8 byte aligned #### see if this is best
	PUSH	QWORD PTR [dbgRSPWK]
	ENDM
	dbgrs	MACRO
	POP	RSP
	POP	RBP
	POP	R15
	POP	R14
	POP	R13
	POP	R12
	POP	R11
	POP	R10
	POP	R9
	POP	R8
	POP	RSI
	POP	RDI
	POP	RDX
	POP	RCX
	POP	RBX
	POP	RAX
	POPFQ
	ENDM


fltsv	macro
	local	myloop
	fwait
	fxsave	fltsave
	fwait
	mov	fltsaversi,rsi
	mov	fltsavercx,rcx

	lea	rsi,fltsave
;mov rcx,512
;call dbgdump
	mov	rcx,64
myloop:
	push	qword ptr [rsi]
	add	rsi,8
	loop	myloop
	mov	rsi,fltsaversi
	mov	rcx,fltsavercx
	endm

fltrs	macro
	local	myloop
	mov	fltsaversi,rsi
	mov	fltsavercx,rcx
	lea	rsi,fltsave+(63*8)
	mov	rcx,64
myloop:
	pop	qword ptr [rsi]
	sub	rsi,8
	loop	myloop
	mov	rsi,fltsaversi
	mov	rcx,fltsavercx
	fwait
	fxrstor	fltsave
	fwait
	endm


	.data

; following are	for debug code

dbgstksav qword	?  ;used for windows system calls to get 16 byte alignment
dbgnumhi qword	?
dbg10tab qword	8000000000000000000,800000000000000000
	qword	80000000000000000,8000000000000000,800000000000000
	qword	80000000000000,8000000000000,800000000000,80000000000
	qword	8000000000,800000000,80000000,8000000,800000,80000,8000,800,80,8

dbgCHAR	db 0,0,0,0,0,0,0,0
dbgLENG	db 0,0,0,0,0,0,0,0
align 8
dbgRSPWK qword 0
dbgRSPWK2 qword	0
dbgmemckad qword 0
dbgiregs qword	   0
dbgcmdl	qword 0
dbgdumpaddr qword 0
dbgdumplen qword 32	    ;length to dump
dumpflt0t dt	 ?
dumpflt1t dt	 ?
dumpflt2t dt	 ?
dumpflt3t dt	 ?
dumpflt4t dt	 ?
dumpflt5t dt	 ?
dumpflt6t dt	 ?
dumpflt7t dt	 ?
dbgmeminfo db 48 DUP(81h)   ; must be 8	byte aligned
dbgmeminfolen equ $-dbgmeminfo
dbgdmpcs db    20 DUP(?)
dbgsnum	db	22 dup(?)
dbgcmd db    512 dup(?)
dbgcmdsize equ $-dbgcmd
;dbgfltemp dq	 0
;dbgfstr db	 512 dup(?)
	align	16
fltsave	dq	64 dup(8181818181818181h)
fltsaversi dq	0
fltsavercx dq	0
stacksave dq	?	;Initial stack pointer
	.code

ifdef linuxenvironment
;linux sys calls: https://syscalls64.paolostivanin.com/
;errors: /usr/src/linux-headers-4.15.0-177/include/uapi/asm-generic/errno-base.h
else
; Windows calling conventions:
; https://docs.microsoft.com/en-us/cpp/build/x64-calling-convention?view=msvc-160#parameter-passing
;
;Integer arguments are passed in registers RCX,	RDX, R8, and R9.
;Floating point	arguments are passed in	XMM0L, XMM1L, XMM2L, and XMM3L.
;16-byte arguments are passed by reference.
;Space is allocated on the call	stack as a shadow store	for
;  callees to save those registers.
;Further parameters are	pushed on the stack.
;These registes, and RAX, R10, R11, XMM4, and XMM5, are	considered volatile.
;A scalar return value that can	fit into 64 bits, including the	__m64 type,
;  is returned through RAX.
;Non-scalar types including floats, doubles, and vector	types such
;  as __m128, __m128i, __m128d are returned in XMM0.
;The state of unused bits in the value returned	in RAX or XMM0 is undefined.
;The x64 ABI considers the registers RAX, RCX, RDX, R8,	R9, R10, R11, and
;  XMM0-XMM5 volatile.
;When present, the upper portions of YMM0-YMM15	and ZMM0-ZMM15 are also
;  volatile.
;On AVX512VL, the ZMM, YMM, and	XMM registers 16-31 are	also volatile.
;Consider volatile registers destroyed on function calls unless	otherwise
;  safety-provable by analysis such as whole program optimization.
;The x64 ABI considers registers RBX, RBP, RDI,	RSI, RSP, R12, R13, R14,
;  R15,	and XMM6-XMM15 nonvolatile.
;They must be saved and	restored by a function that uses them.
;A callee that modifies	any of the fields within FPCSR must restore
;  them	before returning to its	caller.
;GetLastError codes:
;https://docs.microsoft.com/en-us/windows/win32/debug/system-error-codes--0-499-


	dbggle	PROC		;get last error	for windows
	dbgsv
	externdef GetLastError : near
	mov	dbgstksav,rsp
	sub	rsp, 48		    ; must leave 32 bytes space	plus for more shadow parameters
	or	rsp,0fh
	xor	rsp,0fh
	mov	rcx, 0		    ; 1:
	mov	rdx, 0		    ; 2:
	mov	r8d, 0		    ; 3:
	mov	r9, 0		    ; 4:
	mov	QWORD PTR [rsp+32], 0 ;	5: null
	call	GetLastError
	mov	rsp,dbgstksav
	call	dbgs
	db 'GetLastError=',0
	call	dbgprtfi
	call	dbgblnk
	call	dbgprtx
	call	dbgcrlf
	dbgrs
	ret
	dbggle	ENDP
endif

; DBGPRTC prints one character from AL
	dbgprtc	PROC		    ; print the	character in al
	dbgsv
	mov	dbgCHAR,al	    ; character	to be printed


ifdef linuxenvironment
	mov	rdi,1
	lea	rsi,dbgCHAR
	mov	rdx,1
	mov	rax,1		    ;write call
	syscall
else
	includelib kernel32.lib
	includelib user32.lib
	externdef GetStdHandle : near
	externdef WriteFile : near
	externdef VirtualQueryEx : near
	externdef GetCurrentProcess : near
	externdef GetStdHandle : near
	externdef WriteFile : near
	mov	dbgstksav,rsp
	sub	rsp, 48		    ; must leave 32 bytes space	plus for more shadow parameters
	or	rsp,0fh
	xor	rsp,0fh
	mov	ecx, -12	    ; STD_OUTPUT_HANDLE
	call	GetStdHandle

	mov	rcx, rax	    ; 1: handle	was returned in	RAX
	lea	rdx, OFFSET dbgCHAR ; 2: address of char to print
	mov	r8d, 1		    ; 3: number	of chars to print
	lea	r9, QWORD PTR [dbgLENG]	; 4: returns number of bytes written
	mov	QWORD PTR [rsp+32], 0 ;	5: null
	call	WriteFile
	mov	rsp,dbgstksav
endif

	dbgrs
	ret
	dbgprtc	ENDP

; DBGPNTD print	character in DL
	dbgpntd	PROC		    ; print character in DL (Internal use)
	PUSH	RAX
	PUSH	RAX
	MOV	AL,DL
	CALL	dbgprtc
	POP	RAX
	POP	RAX
	RET
	dbgpntd	ENDP

;PRINT AL IN DECIMAL
	dbgprtd	PROC
	dbgsv
	MOV	BL,AL
	MOV	EAX,0
	MOV	AL,BL
	MOV	BL,100
	MOV	RCX,3
DBGlp1:	AND	RAX,255
	DIV	BL
	ADD	AL,30H
	MOV	DL,AL
	MOV	AL,AH
	PUSH	AX
	CALL	dbgpntd
	MOV	AL,BL
	SUB	AH,AH
	MOV	BH,10
	DIV	BH
	MOV	BL,AL
	POP	AX
	LOOP	DBGlp1

	dbgrs
	RET
	dbgprtd	ENDP

;PRINT EAX IN HEX
	dbgprtx4 PROC
;	PUSHFQ
	PUSH	RAX

	ROR	EAX,16
	call	dbgprtx2
	MOV	RAX,QWORD PTR [RSP] ; get copy of RAX from push
	call	dbgprtx2

	POP	RAX
;	POPFQ
	RET
	dbgprtx4 ENDP


;PRINT AX IN HEX
	dbgprtx2 PROC
;	PUSHFQ
	PUSH	RAX

	MOV	AL,AH
	call	dbgprtx1
	MOV	RAX,QWORD PTR [RSP] ; get copy of RAX from push
	call	dbgprtx1

	POP	RAX
;	POPFQ
	RET
	dbgprtx2 ENDP


;PRINT AL IN HEX
	dbgprtx1 PROC
	PUSHFQ
	PUSH	RAX

	ROR	RAX,1
	ROR	RAX,1
	ROR	RAX,1
	ROR	RAX,1
	AND	AL,0FH		    ;isolate 4 bits
	ADD	AL,30H
	CMP	AL,3AH
	JL	DBGlpx1a
	ADD	AL,7
DBGlpx1a:
	CALL	dbgprtc
	MOV	RAX,QWORD PTR [RSP] ; get copy of RAX from push
	AND	AL,0FH		    ;isolate 4 bits
	ADD	AL,30H
	CMP	AL,3AH
	JL	DBGlpx1b
	ADD	AL,7
DBGlpx1b:
	CALL	dbgprtc

	POP	RAX
	POPFQ
	RET
	dbgprtx1 ENDP


;PRINT 10 bytes	at rsi in hex (right to	left)
	dbgprtx10 PROC
	dbgsv
	add	rsi,9
	MOV	RCX,10
DBGlp2:
	mov	al,byte	ptr [rsi]
	ROL	al,1
	ROL	al,1
	ROL	al,1
	ROL	al,1
	AND	AL,0FH		    ;isolate 4 bits
	ADD	AL,30H
	CMP	AL,3AH
	JL	DBGlpx1
	ADD	AL,7
DBGlpx1:
	CALL	dbgprtc

	mov	al,byte	ptr [rsi]
	AND	AL,0FH		    ;isolate 4 bits
	ADD	AL,30H
	CMP	AL,3AH
	JL	DBGlpx2
	ADD	AL,7
DBGlpx2:
	CALL	dbgprtc

	dec	rsi
	LOOP	DBGlp2

	dbgrs
	RET
	dbgprtx10 ENDP


;PRINT RAX IN HEX
	dbgprtx8 PROC
	dbgprtx	PROC
	dbgsv

	MOV	RCX,16
DBGlp2:	PUSH	RCX
	ROL	RAX,1
	ROL	RAX,1
	ROL	RAX,1
	ROL	RAX,1
	PUSH	RAX
	AND	AL,0FH		    ;isolate 4 bits
	ADD	AL,30H
	CMP	AL,3AH
	JL	DBGlpx
	ADD	AL,7
DBGlpx:
	CALL	dbgprtc

	POP	RAX
	POP	RCX
	LOOP	DBGlp2

	dbgrs
	RET
	dbgprtx	ENDP
	dbgprtx8 ENDP

	dbgcrlf	PROC
;PRINT CR-LF
	dbgsv

	MOV	DL,13
	CALL	dbgpntd
	MOV	DL,10
	CALL	dbgpntd

	dbgrs
	RET
	dbgcrlf	ENDP

;PRINT BLANK
	dbgblnk	PROC
	dbgb	PROC
	dbgsv
	MOV	DL,' '
	CALL	dbgpntd
	dbgrs
	RET
	dbgb	ENDP
	dbgblnk	ENDP

; DBGS Print string up to zero after CALL instruction
	dbgs	proc
	PUSHFQ
	PUSH	RAX
	PUSH	RSI
	MOV	RSI,QWORD PTR [RSP+24] ;string loc
DBGSlp:	INC	QWORD PTR [RSP+24]  ;increment return address
	MOV	AL,[RSI]
	INC	RSI
	AND	AL,AL
	JZ	DBGSdone
	CALL	dbgprtc
	JMP	DBGSlp
DBGSdone:
	POP	RSI
	POP	RAX
	POPFQ
	RET
	dbgs	ENDP

; DBGS16 Print string up to zero after CALL instruction
	dbgs16	proc
	PUSHFQ
	PUSH	RAX
	PUSH	RSI
	ADD	QWORD PTR [RSP+24],16
	MOV	RSI,QWORD PTR [RSP+24] ;string loc
DBGS16lp: INC	QWORD PTR [RSP+24]  ;increment return address
	MOV	AL,[RSI]
	INC	RSI
	AND	AL,AL
	JZ	DBGS16done
	CALL	dbgprtc
	JMP	DBGS16lp
DBGS16done:
	POP	RSI
	POP	RAX
	POPFQ
	RET
	dbgs16	ENDP


; DBGREGS display the registers
	dbgregs	proc		    ; dump registers in	hex
	push	rax

	call	dbgs
	db	'RAX=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,rbx
	call	dbgs
	db	'RBX=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,rcx
	call	dbgs
	db	'RCX=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,rdx
	call	dbgs
	db	'RDX=',0
	call	dbgprtx
	call	dbgcrlf

	mov	rax,rsi
	call	dbgs
	db	'RSI=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,rdi
	call	dbgs
	db	'RDI=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,rsp
	pushfq
	add	rax,16		    ;#### fix to get rsp before	this call
	popfq
	call	dbgs
	db	'RSP=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,rbp
	call	dbgs
	db	'RBP=',0
	call	dbgprtx
	call	dbgcrlf

	mov	rax,r8
	call	dbgs
	db	' R8=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,r9
	call	dbgs
	db	' R9=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,r10
	call	dbgs
	db	'R10=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,r11
	call	dbgs
	db	'R11=',0
	call	dbgprtx
	call	dbgcrlf

	mov	rax,r12
	call	dbgs
	db	'R12=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,r13
	call	dbgs
	db	'R13=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,r14
	call	dbgs
	db	'R14=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,r15
	call	dbgs
	db	'R15=',0
	call	dbgprtx
	call	dbgcrlf

;	mov rax,fs	; cannot use fs	or gs without causing os problems
;	call dbgs
;	db 'RFS=',0
;	call dbgprtx
;	call dbgblnk

;	mov rax,gs
;	call dbgs
;	db 'RGS=',0
;	call dbgprtx
;	call dbgblnk

	mov	rax,qword ptr [rsp+8]
	call	dbgs
	db	' IP=',0
	call	dbgprtx
	call	dbgblnk

	pushfq			    ;save to restore after tests
	pushfq			    ;get flags
	pop	rax
	call	dbgs
	db	'flags=',0
	call	dbgprtx

; Show individual flags	here
	test	eax,1		    ;carry
	jz	@f
	call	dbgs
	db	' CF',0
@@:	test	eax,4		    ;parity
	jz	@f
	call	dbgs
	db	' PF',0
@@:	test	eax,16		    ;aux carry
	jz	@f
	call	dbgs
	db	' ACF',0
@@:	test	eax,64		    ;zero
	jz	@f
	call	dbgs
	db	' ZF',0
@@:	test	eax,128		    ;sign
	jz	@f
	call	dbgs
	db	' SF',0
@@:	test	eax,256		    ;trap
	jz	@f
	call	dbgs
	db	' TF',0
@@:	test	eax,512		    ;interrupt enable
	jz	@f
	call	dbgs
	db	' IF',0
@@:	test	eax,1024	    ;direction
	jz	@f
	call	dbgs
	db	' DF',0
@@:	test	eax,2048	    ;overflow
	jz	@f
	call	dbgs
	db	' OF',0
@@:	test	eax,4096+8192	    ;io	privelege level
	jz	@f
	call	dbgs
	db	' IOPL',0
@@:	test	eax,16384	    ;nested flag
	jz	@f
	call	dbgs
	db	' NT',0
@@:	test	rax,65536	    ;resume flag
	jz	@f
	call	dbgs
	db	' RF',0
@@:	test	rax,131072	    ;vm8036 mode
	jz	@f
	call	dbgs
	db	' VM',0
@@:	test	rax,262144	    ;alignment check
	jz	@f
	call	dbgs
	db	' AC',0
@@:	test	rax,524288	    ;virtual interrupt
	jz	@f
	call	dbgs
	db	' VIF',0
@@:	test	rax,1048576	    ;virtual interrup pending
	jz	@f
	call	dbgs
	db	' VIP',0
@@:	test	rax,2097152	    ;ID	flag
	jz	@f
	call	dbgs
	db	' ID',0
@@:	call	dbgcrlf

	popfq

	pop	rax
	ret
	dbgregs	ENDP

; print	just one register
	dbgr11	PROC
	push	rax
	call	dbgs
	db	'r11=',0
	mov	rax,r11
	call	dbgprtx
	call	dbgcrlf
	pop	rax
	ret
	dbgr11	ENDP

	dbgrsp	PROC	;for pat.inc test
	push	rax
	call	dbgs
	db	'rsp=',0
	mov	rax,rsp
	add	rax,16
	sub	rax,stacksave ;####
	neg	rax
	call	dbgprtx
	call	dbgcrlf
	push	rsi
	push	rcx
	mov	rsi,rsp
	add	rsi,32
	mov	rcx,32
	call	dbgdump
	pop	rcx
	pop	rsi
	pop	rax
	ret
	dbgrsp	ENDP

; print	string at RSI up to zero byte
	dbgstrz	PROC
	pushfq
	push	RSI
	push	RCX
	push	RAX
DBGstrzlp:
	mov	al,byte	ptr [rsi]
	cmp	al,0
	jz	@F
	call	dbgprtc
	inc	rsi
	jmp	DBGstrzlp
@@:
	pop	RAX
	pop	RCX
	pop	RSI
	popfq
	ret
	dbgstrz	ENDP

; print	string at RSI for RCX characters
dbgstr	PROC
	jrcxz	DBGstrskip
	pushfq
	push	RSI
	push	RCX
	push	RAX
DBGstrlp:
	mov	al,byte	ptr [rsi]
	call	dbgprtc
	inc	rsi
	loop	DBGstrlp
	pop	RAX
	pop	RCX
	pop	RSI
	popfq
DBGstrskip: ret
	dbgstr	ENDP

; show who called the routine in which this call is placed
	dbgfrom	proc private
	push	rax
	mov	rax,[rsp+16]	    ;assumes return address was	most recent in stack
	call	dbgs
	db	'Called	from ',0
	call	dbgprtx
	call	dbgcrlf
	pop	rax
	ret
	dbgfrom	endp

;Print stack contents starting at rsp for rcx bytes
	dbgstack PROC
	push	rsi
	mov	rsi,rsp
	add	rsi,16
	call	dbgdump
	pop	rsi
	ret
	dbgstack endp

;Print memory dump
	dbgdump	PROC	private
	dbgsv
; rdi -	pointer	character string build-up area
; rsi -	pointer	to data	to dump
; rcx -	length of data to dump
	mov	dbgdumpaddr,rsi
	cmp	rsi,0
	jz	dbgdumpbad	    ; zero address not allowed
	call	dbgckad		    ; check if address is ok
	cmp	rax,rsi
	ja	dbgdumpck1	    ; starting address is less than limit
;call dbgprtx8
;call dbgs
;db '=rax dbg badck1',13,10,0
	jmp	dbgdumpbad
dbgdumpck1:
	sub	rax,rcx
	cmp	rax,rsi
	ja	dbgbyte0	    ; addres in	range
	add	rax,rcx		    ; restore rax
	mov	rcx,rax
	sub	rcx,rsi		    ; rcx adjusted to what is dumpable
;call dbgs
;db 'dbg badck2',13,10,0
dbgbyte0:
	mov	rdi,offset dbgdmpcs ;work area to assemble character version on	right

	mov	rax,rsi
	call	dbgcrlf
	call	dbgprtx		    ;First print the starting address
	call	dbgs
	db	': ',0

; Put source and length	on stack
	push	rax		    ;will be [rsp+8] (address)
	push	rcx		    ;will be [rsp+0] (length)
; Produce proper number	of blanks to recover alignment
	and	rax,0Fh
	mov	rcx,rax
	jrcxz	dbgnopad1	    ;jump if no	leading	padding	needed
dbgpad1lp:
	call	dbgblnk		    ;two blanks	for hex	region
	call	dbgblnk
	mov	byte ptr [rdi],' '
	inc	rdi		    ;one blank for character region
	test	rcx,03h
	jnz	dbgnobet
	call	dbgblnk		    ;extra blank after each 4 bytes
	mov	byte ptr [rdi],' '
	inc	rdi
dbgnobet:
	loop	dbgpad1lp
dbgnopad1:
	pop	rcx		    ;(length)
	pop	rax		    ;(address)

; now process the data for one print line
dbgdmplp1:
	call	dbgdmpw		    ;show a word in hex	and char (blanks if ecx=0)

	call	dbgblnk
	mov	byte ptr [rdi],' '
	inc	rdi

	test	rsi,0Fh
	jnz	dbgdmplp1	    ;dump until	end of 16 bytes
	call	dbgdumpr	    ;go	print character	version	on right side
	cmp	rcx,0
	jnz	dbgbyte0	    ;do	another	line
	call	dbgcrlf
	jmp	dbgdumpdone

dbgdumpbad:
	call	dbgs
	db	'attempt to dump at bad	address:',0
	mov	rax,dbgdumpaddr
	call	dbgprtx8
	call	dbgblnk
	call	dbgs
	db	' limit:',0
	mov	rax,dbgmemckad
	call	dbgprtx8
	call	dbgcrlf
dbgdumpdone:
	dbgrs
	ret
	dbgdump	ENDP
	dbgdmpw	PROC
	jecxz	dbgdmp0bl1
	jmp	dbgdmpn0
dbgdmp0bl:
	test	rsi,00000003h
	jz	dbgdmp0
dbgdmp0bl1:
	call	dbgblnk
	call	dbgblnk
	inc	rsi
	mov	byte ptr [rdi],' '
	inc	rdi
	jmp	dbgdmp0bl	    ;do	blanks until this word space filled
dbgdmp0: ret
dbgdmpn0:
	mov	al,0bdh
;######	call	dbgchkaddr
;	jz	dbgdmpokad
;	mov	rcx,1
;	jmp	dbgdmpbadad
dbgdmpokad:
	mov	al,byte	ptr [rsi]
dbgdmpbadad:
	call	dbgprtx1	    ;print two hex digits
	inc	rsi
	cmp	al,20h
	jnb	dbgchnlow
	mov	al,'.'		    ;put . for characters below	ascii blank
dbgchnlow:
	cmp	al,80h
	jc	dbgchnhi
	mov	al,'.'		    ;put . fir chars above 7Fx
dbgchnhi:
	mov	[rdi],al
	inc	rdi
	loop	dbgbyte2
	jmp	dbgdmp0bl
dbgbyte2: test	  rsi,00000003h
	jz	dbgdmp0		    ;jump if at	end of word
	jmp	dbgdmpn0
	dbgdmpw	ENDP

; pring	byte as	printable character otherwise "."
	dbgdumpr proc
	push	rcx
	mov	rdi,offset dbgdmpcs
	mov	rcx,19
dbgchprtlp:
	mov	al,[rdi]
	call	dbgprtc
	inc	rdi
	loop	dbgchprtlp
	pop	rcx
	ret
	dbgdumpr ENDP

; convert 64 bit binary	number in RAX to signed	decimal	digits
; result at dbgsnum
	dbgitos	proc
	dbgsv

	mov	dbgsnum,'-'
	and	rax,rax
	jl	dbgnoinv	    ;jump if number is negative
	not	rax		    ;twos complement because negative can be 1 larger
	add	rax,1
	mov	dbgsnum,'+'	    ;note that it was positive
dbgnoinv:
	mov	[dbgnumhi],rax	    ;save the negative number

	lea	rsi,dbg10tab	    ; address into tens	table
	lea	rdi,[dbgsnum+1]	    ;address to	put first digit	at
	mov	rcx,19		    ;number of digits max
	mov	rax,0800h	    ;max possible for leftmost digit
	jmp	dbgnumst
dbgnumlp2:
	mov	rax,0800h	    ;max binary	for any	other digit
dbgnumst:
	mov	rdx,[rsi]	    ;load from dbg10tab
dbgnumlp:
	add	[dbgnumhi],rdx	    ;see if this contributes to	digit
	jg	dbgdigno	    ;jump if test number was smaller
dbgdigok:
	add	al,ah		    ;add ah to show digit contribution from dbg10tab
dbgnumnx:
	shr	ah,1		    ;divide by 2
	jz	dbgdigdn	    ;jump if done with this digit
	shr	rdx,1		    ;divide the	dbg10tab values	by 2
	jmp	dbgnumlp
dbgdigno:
	sub	[dbgnumhi],rdx	    ;undo add from above
	jmp	dbgnumnx	    ;move to next digit
dbgdigdn:
	add	rsi,8		    ;go	to next	number in dbg10tab
	add	al,'0'		    ;make digit	into character
	mov	[rdi],al
	inc	rdi
	loop	dbgnumlp2	    ;jump if more digits to do

	dbgrs
	ret
	dbgitos	endp

	dbgprtfi proc
	dbgsv
	call	dbgitos
	cmp	dbgsnum,'-'
	jne	dbgnn		    ;don't print positive sign
	mov	al,dbgsnum
	call	dbgprtc
dbgnn:	lea	rsi,[dbgsnum+1]
	mov	rcx,19
dbglzck: cmp	 byte ptr [rsi],'0' ;check for leading zeros
	jne	dbgnlz
	inc	rsi
	loop	dbglzck
	mov	al,'0'		    ;number was	all zero
	call	dbgprtc
	jmp	dbgprtfie	    ;done
dbgnlz:	mov	al,[rsi]
	call	dbgprtc
	inc	rsi
	loop	dbgnlz
dbgprtfie:
	dbgrs
	ret
	dbgprtfi endp

;==========================================
; dbgckad Check	if address in rsi is valid (so can avoid an exception on read)
; input: address in rsi	    zero not allowed
; returns: address in rax.  If rsi>=rax	then address might give	execption
	dbgckad	proc
	dbgsv

	mov	dbgmemckad,0	    ;default is	failure	address
	cmp	rsi,dbgmemckad	    ;zero address not allowed
	je	dbgmemckbad
ifdef linuxenvironment
	mov	rdi,rsi
	shr	rdi,12
	shl	rdi,12		    ;round down	to 4096	page size
	mov	rsi,4096	    ;page size (apparently not available from a	syscall)
	mov	rdx,1
	mov	rax,01Ah	    ;sys_msync call
	syscall
;call dbgs
;db 'at	ckad lnx sys_msync return',13,10,0
;call dbgregs
	cmp	rax,0		    ;address ok	if 0
	jne	dbgmemckbad
	dec	dbgmemckad	    ;don't know	how to find highest ok address #### just make it all F
else
	externdef VirtualQueryEx : near
	externdef GetCurrentProcess : near
	mov	dbgstksav,rsp
	sub	rsp, 88		    ; must leave 32 bytes space	plus for more shadow parameters
	or	rsp,0fh
	xor	rsp,0fh
	call	GetCurrentProcess   ; process handle returned in rax

	mov	r9,  dbgmeminfolen  ; 4. argument: r9  = memory	info area length
	lea	r8, [dbgmeminfo]    ; 3. argument: r8  = PMEMORY_BASIC_INFORMATION area, 8 byte	aligned
	mov	rdx, rsi	    ; 2. argument: edx = test address
; call dbgs
; db 'dbg Address being	checked=',0
; push rax
; mov rax,rdx
; call dbgprtx8
; pop rax
; call dbgcrlf
; call dbgprtx8
; call dbgs
; db '=rax is process handle dbg',13,10,0
; call dbgcrlf
	mov	rcx, rax	    ; 1. argument: rcx = current process handle
; call dbgs
; db 'dbg regs before VirtualQueryEx call:',13,10,0
; call dbgregs
	call	VirtualQueryEx
	mov	rsp,dbgstksav

;call dbgs
;db 'at	ckad win return	from VirtualQueryEx in dbg regs=',13,10,0
;call dbgregs

	cmp	rax,30h		    ;check if result looked ok
	jne	dbgmemckbad
	cmp	dword ptr dbgmeminfo+32,1000h ;check if	state is what we want
	jne	dbgmemckbad
	mov	rax,qword ptr dbgmeminfo ;base address
	add	rax,qword ptr dbgmeminfo+24 ;add region	size to	get last+1 of region
	mov	dbgmemckad,rax

endif
dbgmemckbad:
	dbgrs
	mov	rax,dbgmemckad
	ret
	dbgckad	endp

; =======================================================
; DBGI - interactive debug
dbgi	proc
	mov	dbgiregs,rsp	    ;remember pointer to regs
	dbgsv
	call	dbgs
	db	13,10,'Interactive Debug, ? for	help',13,10,0

dbgi_new:
ifdef linuxenvironment
	mov	rdi,1		    ;standard handle?
	lea	rsi,dbgcmd
	mov	rdx,0
	mov	edx,dbgcmdsize
	mov	rax,0		    ;sys_read
	syscall
	mov	rcx,rax		    ;number of bytes that were read
	mov	dbgcmdl,rax
	lea	rsi,dbgcmd
 call dbgstr
else
	mov	dbgstksav,rsp
	sub	rsp, 48		    ; must leave 32 bytes space	plus for more shadow parameters
	or	rsp,0fh
	xor	rsp,0fh
	mov	ecx,-10		    ;STD_INPUT_HANDLE
	externdef ReadConsoleA : near
	call	GetStdHandle
	mov	rcx,rax		    ;handle
	lea	rdx,dbgcmd
	mov	r8d,dbgcmdsize
	lea	r9,qword ptr dbgcmdl
	mov	qword ptr [rsp+32],0
	call	ReadConsoleA
	mov	rsp,dbgstksav
;mov rcx,dbgcmdl
;lea rsi,dbgcmd
;call dbgstr
endif


;***************************
; Parse	debug command
	MOV	RCX,dbgcmdl	; Length of command
	LEA	RSI,dbgcmd	; Command address
	call	dbg_sb		; Span any blanks
	jrcxz	dbgi_new	; Go read a new	command
	cmp	byte ptr [rsi],'q'	;Quit command?
	je	dbgi_quit
	cmp	byte ptr [rsi],'Q'	;Quit command?
	je	dbgi_quit
	cmp	byte ptr [rsi],'d'	;Display command?
	je	dbgi_disp
	cmp	byte ptr [rsi],'D'	;Display command?
	je	dbgi_disp
;	cmp	byte ptr [rsi],'2'	;Dual output toggle?
;	je	dbgi_dual
	cmp	byte ptr [rsi],'r'	;Display registers
	je	dbgi_regs
	cmp	byte ptr [rsi],'R'	;Display registers
	je	dbgi_regs
	cmp	byte ptr [rsi],'?'	;Show command list
	je	dbgi_help
	cmp	byte ptr [rsi],'x'	;Exit program
	je	dbgi_exitpgm
	cmp	byte ptr [rsi],'X'	;Exit program
	je	dbgi_exitpgm
;	cmp	byte ptr [rsi],'s'	;Search	command
;	je	dbgi_search
;	cmp	byte ptr [rsi],'S'	;Search	command
;	je	dbgi_search
;	cmp	byte ptr [rsi],'w'	;Whereis command
;	je	dbgi_whereis
;	cmp	byte ptr [rsi],'W'	;Whereis command
;	je	dbgi_whereis
;	cmp	byte ptr [rsi],'?'	;Help command
;	je	dbgi_help
; Put check for	new commands here


dbgi_error1:
	call	dbgs
	db	'Unknown debug command,	use ? for help',13,10,0
	jmp	dbgi_new
dbgi_quit:
	dbgrs
	ret

dbgi_exitpgm:
	call	dbgexit		;exit this process

; Put code for commands	here ==================================


;***************************************
; Display command
dbgi_disp:
	inc	rsi
	dec	rcx
	jrcxz	dbgi_disp_na
	call	dbg_sb		;skip blanks
	jrcxz	dbgi_disp_na
	call	dbg_hn		;get hex number	into eax
	jc	dbgi_disp_na	;jump if no hex	number there
	mov	dbgdumpaddr,rax
	call	dbg_sb		;skip blanks
	mov	rax,dbgdumplen	;prior or initial display length
	call	dbg_hn		;get hex number	for length
	jc	dbgi_disp_na
	mov	dbgdumplen,rax	;set new length
	call	dbg_sb
	jecxz	dbgi_disp_na	;jump if at end	of command
	call	dbgs
	db	'Extra parameters ignored on D command',13,10,0
dbgi_disp_na:
	mov	rsi,dbgdumpaddr
	mov	rcx,rsi
	add	rcx,dbgdumplen	;length	to dump
	add	rcx,15		;add for rounding
	and	rcx,0fffffff0h
	mov	dbgdumpaddr,rcx	;address for next time
	sub	rcx,rsi		;length	for this time
	call	dbgdump		;fake for now
	jmp	dbgi_new


;***************************************
; Display command list
dbgi_help:
	call	dbgs
	db	'? => Show this	list of	commands',13,10,0
;	call	dbgs
;	db	'2 => Dual output toggle (adds STDERR output to	STD output)',13,10,0
	call	dbgs
	db	'D addr	len => Display memory at addr for length len',13,10,0
	call	dbgs
	db	'Q => Quit',13,10,0
	call	dbgs
	db	'X => Exit process',13,10,0
	call	dbgs
	db	'R => Display registers',13,10,0
;	call	dbgs
;	db	'S hhhhhhhh => Search for 4 byte hex string from dump point',13,10,0
	jmp	dbgi_new

;***************************************
; Display registers
; FLAGS	  [dbgiregs-8]
; RAX	  [dbgiregs-16]
; RBX	  [dbgiregs-24]
; RCX	  [dbgiregs-32]
; RDX	  [dbgiregs-40]
; RDI	  [dbgiregs-48]
; RSI	  [dbgiregs-56]
; R8	  [dbgiregs-64]
; R9	  [dbgiregs-72]
; R10	  [dbgiregs-80]
; R11	  [dbgiregs-88]
; R12	  [dbgiregs-96]
; R13	  [dbgiregs-104]
; R14	  [dbgiregs-112]
; R15	  [dbgiregs-120]
; RBP	  [dbgiregs-128]
dbgi_regs:
	mov	rdi,dbgiregs

	call	dbgs
	db	'RAX=',0
	mov	rax,[rdi-16]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'RBX=',0
	mov	rax,[rdi-24]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'RCX=',0
	mov	rax,[rdi-32]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'RDX=',0
	mov	rax,[rdi-40]
	call	dbgprtx
	call	dbgcrlf

	call	dbgs
	db	'RSI=',0
	mov	rax,[rdi-56]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'RDI=',0
	mov	rax,[rdi-48]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'RSP=',0
	mov	rax,rdi
	add	rax,8		    ;#### fix to get rsp before	this call
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'RBP=',0
	mov	rax,[rdi-128]
	call	dbgprtx
	call	dbgcrlf

	call	dbgs
	db	' R8=',0
	mov	rax,[rdi-64]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	' R9=',0
	mov	rax,[rdi-72]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'R10=',0
	mov	rax,[rdi-80]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'R11=',0
	mov	rax,[rdi-88]
	call	dbgprtx
	call	dbgcrlf

	call	dbgs
	db	'R12=',0
	mov	rax,[rdi-96]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'R13=',0
	mov	rax,[rdi-104]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'R14=',0
	mov	rax,[rdi-112]
	call	dbgprtx
	call	dbgblnk

	call	dbgs
	db	'R15=',0
	mov	rax,[rdi-120]
	call	dbgprtx
	call	dbgcrlf

	mov	rax,qword ptr [rdi]
	call	dbgs
	db	' IP=',0
	call	dbgprtx
	call	dbgblnk

	mov	rax,[rdi-8]
	call	dbgs
	db	'flags=',0
	call	dbgprtx

; Show individual flags	here
	test	eax,1		    ;carry
	jz	@f
	call	dbgs
	db	' CF',0
@@:	test	eax,4		    ;parity
	jz	@f
	call	dbgs
	db	' PF',0
@@:	test	eax,16		    ;aux carry
	jz	@f
	call	dbgs
	db	' ACF',0
@@:	test	eax,64		    ;zero
	jz	@f
	call	dbgs
	db	' ZF',0
@@:	test	eax,128		    ;sign
	jz	@f
	call	dbgs
	db	' SF',0
@@:	test	eax,256		    ;trap
	jz	@f
	call	dbgs
	db	' TF',0
@@:	test	eax,512		    ;interrupt enable
	jz	@f
	call	dbgs
	db	' IF',0
@@:	test	eax,1024	    ;direction
	jz	@f
	call	dbgs
	db	' DF',0
@@:	test	eax,2048	    ;overflow
	jz	@f
	call	dbgs
	db	' OF',0
@@:	test	eax,4096+8192	    ;io	privelege level
	jz	@f
	call	dbgs
	db	' IOPL',0
@@:	test	eax,16384	    ;nested flag
	jz	@f
	call	dbgs
	db	' NT',0
@@:	test	rax,65536	    ;resume flag
	jz	@f
	call	dbgs
	db	' RF',0
@@:	test	rax,131072	    ;vm8036 mode
	jz	@f
	call	dbgs
	db	' VM',0
@@:	test	rax,262144	    ;alignment check
	jz	@f
	call	dbgs
	db	' AC',0
@@:	test	rax,524288	    ;virtual interrupt
	jz	@f
	call	dbgs
	db	' VIF',0
@@:	test	rax,1048576	    ;virtual interrup pending
	jz	@f
	call	dbgs
	db	' VIP',0
@@:	test	rax,2097152	    ;ID	flag
	jz	@f
	call	dbgs
	db	' ID',0
@@:	call	dbgcrlf
	jmp	dbgi_new

;==============================================================

dbgi	endp


; Skip blanks and tabs (white space)
dbg_sb	proc
	jrcxz	dbg_sb_ret
dbg_sb_lp:
	cmp	byte ptr [rsi],' '	;check for blank
	je	dbg_sb_skip
	cmp	byte ptr [rsi],9	;check for tab
	je	dbg_sb_skip
	cmp	byte ptr [rsi],13	;check for carriage return
	je	dbg_sb_skip
	cmp	byte ptr [rsi],10	;check for line	feed
	je	dbg_sb_skip
dbg_sb_ret: ret
dbg_sb_skip:
	inc	rsi
	loop	dbg_sb_lp
	ret
dbg_sb	endp

; DBGEXIT exit this process
dbgexit	proc
	call	dbgs
	db	'Exiting Process',13,10,0
ifdef linuxenvironment
	mov	rdi,0		    ;return code
	mov	rax,60		    ;exit call
	syscall			    ;exit program
else
	externdef ExitProcess :	near
	mov	rcx, 0		    ; return code
	call	ExitProcess
endif
dbgexit	endp


; Read in hex number into RAX from RSI RCX, set	carry if no number
; RAX is original value	if no number found
dbg_hn	proc	private
	push	rbx
	push	rdx
	jecxz	dbg_hn_fail
	mov	rdx,rcx		;save original remaining length
	xor	rax,rax
	xor	rbx,rbx
dbg_hn_lp:
	mov	bl,[rsi]
	cmp	bl,' '		;blank?
	jz	dbg_hn_dn
	cmp	bl,9		;tab?
	jz	dbg_hn_dn
	sub	bl,'0'
	cmp	bl,9
	jna	dbg_hn_ad	;jump if not alpha digit
	sub	bl,'A'-'0'-10
	cmp	bl,10
	jb	dbg_hn_dn
	cmp	bl,15
	jna	dbg_hn_ad	;jump if not lower case	alpha
	sub	bl,'a'-'A'
	cmp	bl,10
	jb	dbg_hn_dn
	cmp	bl,15
	jna	dbg_hn_ad
	cmp	rcx,rdx
	je	dbg_hn_fail	;jump to failure if no digits
	jmp	dbg_hn_dn
dbg_hn_ad:
	shl	rax,4
	add	rax,rbx
	inc	rsi
	loop	dbg_hn_lp
dbg_hn_dn:
	pop	rdx
	pop	rbx
	clc				;clear carry flag
	ret
dbg_hn_fail:
	pop	rdx
	pop	rbx
	stc				;set carry flag
	ret
dbg_hn	endp

;============================================
dbgistr	macro	a,b,c,d,e,f
	call	dbgs
	db '&a&	&b& &c&	&d& &e&	&f&',13,10,0
	endm

;============================================
dbgustr	macro	a
	call	dbgs
	db &a&,13,10,0
	endm

;============================================
dbgqword macro	a,b
	call	dbgs
	db '&b&	&a&=',0
	push	rax
	mov	rax,a
	call	dbgprtx
	call	dbgcrlf
	pop	rax
	endm

;============================================
; show character string	in structure at	address	x
showstr	macro	x
	local	place
	push	rax
	mov	rax,x
	call	dbgprtx
	call	dbgs
	db	' str is "',0
	push	rsi
	push	rcx
	mov	rcx,[rax+VOFFSET]
	shr	rcx,8
	lea	rsi,[rax+DESCR*4]
	call	dbgstr
	call	dbgs
	db	'"',0
	mov	rsi,rax
	mov	rcx,4*DESCR
	call	dbgdump
	mov	rsi,[rax+32]
	cmp	rsi,0
	jz	place
	call	dbgdump
place:
	pop	rcx
	pop	rsi
	pop	rax
	endm

;============================================
; show block structure at address x
showblk	macro	x
	local	isstring,showdone,notzer,skipit
	push	rsi
	push	rcx
	IFDIF	<x>,<rsi>
	  mov	  rsi,x
	ENDIF
	call	dbgs
	db	'Block is:',13,10,0
	cmp	rsi,0
	jnz	notzer
	call	dbgs
	db 'zero in &x&',13,10,0
	jmp	skipit
notzer:	mov	rcx,[rsi+VOFFSET]
	test	cl,STTL
	jnz	isstring
	shr	rcx,8
	add	rcx,DESCR
	jmp	showdone
isstring: shr	rcx,8
	add	rcx,64
showdone: call	dbgdump
skipit:	pop	rcx
	pop	rsi
	endm

;============================================
; DEBUG	counter	useful for checking things after a specific count
dbgcount macro	x,mytxt
	local notyet
	inc	DBGcounter
	IFDIF	<x>,<>
	  push rax
	  mov rax,x
	  cmp rax,DBGcounter
	  ja notyet
	  mov rax,DBGcounter
	  call dbgprtfi
	  call dbgs
	  db '=DBGcounter &mytxt&',13,10,0
notyet:
	  pop  rax
	ENDIF
	endm

dbgcntmsg macro	n,mytxt
	local notyet
	cmp DBGcounter,n
	jb notyet
	push rax
	mov rax,DBGcounter
	call dbgprtfi
	call dbgs
	db '=DBGcounter	looking	at &mytxt&',13,10,0
	pop rax
notyet:
	endm


; ============================================
showstk	macro
	push rax
	mov rax,rsp
	add rax,8
	call dbgprtx
	call dbgs
	db '->',0
	mov rax,[rsp+8]
	call dbgprtx
	call dbgs
	db '=first qword in stack',13,10,0
	push rsi
	push rcx
	mov  rsi,rax
	mov  rcx,32
	call dbgdump
	pop  rcx
	pop  rsi
	pop rax
	endm

; ============================================
showreg	macro regis
	push rax
	mov rax,regis
	IFIDN <regis>,<rsp>
	add rax,8
	ENDIF
	call dbgprtx
	call dbgs
	db '=value of &regis&',13,10,0
	pop rax
	endm
; ============================================
showreg4 macro regis
	push rax
	mov eax,regis
	call dbgprtx4
	call dbgs
	db '=value of &regis&',13,10,0
	pop rax
	endm

; ============================================
showregi macro regis
	push rax
	mov rax,regis
	call dbgprtfi
	call dbgs
	db '=value of &regis&',13,10,0
	pop rax
	endm

; ============================================
dumpsome macro regis
	push rsi
	push rcx
	call dbgs
	db 'showing what &regis& points	to:',13,10,0
	mov rsi,regis
	mov rcx,96
	call dbgdump
	pop rcx
	pop rsi
	endm

; ============================================
dbgfregs proc
	dbgsv
	fltsv

	fstp	dumpflt0t
	fstp	dumpflt1t
	fstp	dumpflt2t
	fstp	dumpflt3t
	fstp	dumpflt4t
	fstp	dumpflt5t
	fstp	dumpflt6t
	fstp	dumpflt7t

	lea	rsi,dumpflt0t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(0)'

	lea	rsi,dumpflt1t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(1)'

	lea	rsi,dumpflt2t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(2)'

	lea	rsi,dumpflt3t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(3)'

	lea	rsi,dumpflt4t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(4)'

	lea	rsi,dumpflt5t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(5)'

	lea	rsi,dumpflt6t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(6)'

	lea	rsi,dumpflt7t
	call	dbgprtx10
	call	dbgblnk
	dbgustr	'=st(7)'

	call	dbgcrlf

	fld	dumpflt7t
	fld	dumpflt6t
	fld	dumpflt5t
	fld	dumpflt4t
	fld	dumpflt3t
	fld	dumpflt2t
	fld	dumpflt1t
	fld	dumpflt0t
	fltrs
	dbgrs
	ret
dbgfregs endp

